<?php
// gameapp/world_map.php
// Renderer for procedural map data, expects generator utilities from map_generation.php.

require_once __DIR__ . '/map_generation.php';

if (!function_exists('apw_render_map_html')) {
    /**
     * Render HTML grid for the generated map, including nations and cities when available.
     */
    function apw_render_map_html(array $map): string {
        $palette = [
            'water'    => '#3c74b5',
            'coast'    => '#6ea5d8',
            'plains'   => '#9abd5b',
            'forest'   => '#4f7b42',
            'wetland'  => '#87b4a2',
            'desert'   => '#d9c07a',
            'hill'     => '#b59e68',
            'mountain' => '#8a8a8a',
        ];

        $nation_colors = [];
        if (!empty($map['nations'])) {
            foreach ($map['nations'] as $nation) {
                $nation_colors[$nation['id']] = $nation['color'];
            }
        }

        $legend_items = '';
        foreach ($palette as $name => $color) {
            $legend_items .= '<span class="legend-item"><span style="background:'
                . htmlspecialchars($color) . '"></span>'
                . htmlspecialchars(ucfirst($name)) . '</span>';
        }
        if ($nation_colors) {
            $legend_items .= '<span class="legend-item"><span class="legend-nation"></span>Nation borders</span>';
            $legend_items .= '<span class="legend-item"><span class="legend-city"></span>City center</span>';
        }

        $cells = '';
        for ($y = 0; $y < $map['height']; $y++) {
            for ($x = 0; $x < $map['width']; $x++) {
                $tile = $map['tiles'][$y][$x];
                $terrain = $tile['terrain'];
                $color = $palette[$terrain] ?? '#888';
                $border = (!empty($tile['border']) && isset($nation_colors[$tile['nation_id']]))
                    ? 'border: 1px solid ' . htmlspecialchars($nation_colors[$tile['nation_id']]) . ';'
                    : '';
                $city_marker = $tile['city'] ? 'data-city="1"' : '';
                $nation_attr = $tile['nation_id'] ? 'data-nation="' . (int)$tile['nation_id'] . '"' : '';
                $title = 'Tile ' . $x . ',' . $y
                    . ' • ' . ucfirst($terrain)
                    . ' • Height ' . $tile['height']
                    . ' • Moisture ' . $tile['moisture']
                    . ' • Resources ' . $tile['resource_score'];
                if (!empty($tile['population'])) {
                    $title .= ' • Pop ' . $tile['population'];
                }
                if ($tile['city']) {
                    $title .= ' • City ' . $tile['city'];
                }

                $cells .= '<div class="map-cell terrain-' . htmlspecialchars($terrain)
                    . '" style="background:' . htmlspecialchars($color) . ';' . $border
                    . '" title="' . htmlspecialchars($title) . '" ' . $nation_attr . ' ' . $city_marker
                    . '></div>';
            }
        }

        $grid_style = 'grid-template-columns: repeat(' . (int)$map['width'] . ', 1fr);'
            . 'grid-template-rows: repeat(' . (int)$map['height'] . ', 1fr);';

        return '<div class="map-preview">'
            . '<div class="map-legend">' . $legend_items . '</div>'
            . '<div class="map-grid" style="' . $grid_style . '">' . $cells . '</div>'
            . '</div>';
    }
}
?>
