<?php
// gameapp/teacher_tab_games.php
require_once __DIR__ . '/world_map.php';

$resolution_cap = 'standard';
$acct_name = strtolower($teacher['account_name'] ?? '');
if (str_contains($acct_name, 'district') || str_contains($acct_name, 'enterprise')) {
    $resolution_cap = 'enterprise';
} elseif (str_contains($acct_name, 'advanced') || str_contains($acct_name, 'pro')) {
    $resolution_cap = 'advanced';
}
?>
<section id="tab-games" class="tab-pane">
    <div class="card">
        <div class="card-header-row">
            <div>
                <h2>Games</h2>
                <p class="meta">
                    Games used: <strong><?php echo $total_games; ?></strong> /
                    <strong><?php echo $max_games; ?></strong>
                </p>
                <?php if ($delete_error): ?>
                    <p class="meta" style="color:#b33;">
                        Games that already have recorded turns cannot be deleted.
                    </p>
                <?php endif; ?>
            </div>
            <?php if ($total_games < $max_games): ?>
                <button type="button" class="btn btn-primary" onclick="toggleCreateGame()">
                    Create new game
                </button>
            <?php endif; ?>
        </div>

        <p class="meta">Click a game name to open its design / edit interface.</p>

        <?php if (!$games): ?>
            <p>No games yet. Click the + button to create your first game.</p>
        <?php else: ?>
            <ul class="game-list">
                <?php foreach ($games as $g): ?>
                    <li>
                        <div style="display:flex;justify-content:space-between;align-items:center;gap:0.5rem;">
                            <div>
                                <a href="teacher_dashboard.php?tab=games&game=<?php echo (int)$g['id']; ?>">
                                    <span class="game-name"><?php echo htmlspecialchars($g['name']); ?></span>
                                </a>
                                <span class="game-meta">
                                    Code: <code><?php echo htmlspecialchars($g['game_code']); ?></code> ·
                                    Map: <?php echo htmlspecialchars($g['map_type']); ?>/<?php echo htmlspecialchars($g['map_size']); ?> ·
                                    Nations: <?php echo (int)$g['num_nations']; ?> ·
                                    Turn: <?php echo isset($g['current_turn']) ? (int)$g['current_turn'] : 1; ?> ·
                                    Mode: <?php echo htmlspecialchars($g['turn_mode'] ?? 'manual'); ?>
                                </span>
                            </div>
                            <div>
                                <?php
                                $stmt = $pdo->prepare('SELECT COUNT(*) AS cnt FROM game_turns WHERE game_id = ?');
                                $stmt->execute([(int)$g['id']]);
                                $row = $stmt->fetch();
                                $turns = (int)$row['cnt'];
                                ?>
                                <?php if ($turns === 0): ?>
                                    <form method="post" class="inline-form"
                                          onsubmit="return confirm('Delete this game? This will remove nations and all configuration.');">
                                        <input type="hidden" name="delete_game" value="1">
                                        <input type="hidden" name="game_id" value="<?php echo (int)$g['id']; ?>">
                                        <button type="submit" class="btn btn-danger btn-small">Delete</button>
                                    </form>
                                <?php else: ?>
                                    <span class="meta">Has turns – cannot delete</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </li>
                <?php endforeach; ?>
            </ul>
        <?php endif; ?>

        <div class="card-subsection">
            <h3>Analytics API</h3>
            <p class="meta">
                Use <code>api_game_overview.php?game_id=ID</code> to fetch game structure, nations,
                assignments, and basic turn metrics for dashboards and graphs.
            </p>
            <?php if ($selected_game): ?>
                <p class="meta">
                    Selected game:
                    <code>api_game_overview.php?game_id=<?php echo (int)$selected_game['id']; ?></code>
                </p>
            <?php endif; ?>
        </div>
    </div>

    <div class="card">
        <div class="card-header-row">
            <div>
                <h2>Students (quick view)</h2>
                <p class="meta">
                    Seats used: <strong><?php echo $used_seats; ?></strong> /
                    <strong><?php echo $max_students; ?></strong>
                    &nbsp;•&nbsp; Empty seats: <strong><?php echo $empty_seats; ?></strong>
                </p>
            </div>
            <a class="btn btn-ghost" href="teacher_dashboard.php?tab=students">Full student view</a>
        </div>

        <?php if ($empty_seats > 0): ?>
            <form method="post" class="inline-form">
                <input type="hidden" name="create_student" value="1">
                <label style="margin-right:0.5rem;">
                    <span class="meta">New student label</span>
                    <input type="text" name="label" placeholder="Name or alias">
                </label>
                <button type="submit" class="btn btn-primary">Create student</button>
            </form>
        <?php endif; ?>

        <?php if (!$students): ?>
            <p class="meta">No students yet.</p>
        <?php else: ?>
            <div class="table-wrapper" style="margin-top:0.75rem;">
                <table>
                    <thead>
                    <tr>
                        <th>Label</th>
                        <th>Code</th>
                        <th>Game</th>
                        <th>Nation</th>
                        <th>Actions</th>
                        <th>Delete</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($students as $s): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($s['label'] ?? ''); ?></td>
                            <td>
                                <code><?php echo htmlspecialchars($s['student_code']); ?></code><br>
                                <a href="game.php?s=<?php echo urlencode($s['student_code']); ?>" target="_blank">
                                    Open view
                                </a>
                            </td>
                            <td>
                                <form method="post" class="inline-form">
                                    <input type="hidden" name="student_action" value="assign_game_nation">
                                    <input type="hidden" name="student_id"
                                           value="<?php echo (int)$s['id']; ?>">

                                    <select name="student_game" class="student-game-select">
                                        <option value="0">No game</option>
                                        <?php foreach ($games as $g): ?>
                                            <option value="<?php echo (int)$g['id']; ?>"
                                                <?php if ((int)$s['game_id'] === (int)$g['id']) echo 'selected'; ?>>
                                                <?php echo htmlspecialchars($g['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                            </td>
                            <td>
                                    <select name="student_nation" class="student-nation-select">
                                        <option value="0">No nation</option>
                                        <?php
                                        if (!empty($s['game_id']) && isset($game_nations[$s['game_id']])) {
                                            foreach ($game_nations[$s['game_id']] as $n) {
                                                $sel = ((int)$s['nation_id'] === (int)$n['id']) ? 'selected' : '';
                                                echo '<option value="' . (int)$n['id'] . '" ' . $sel . '>' .
                                                     htmlspecialchars($n['name']) . '</option>';
                                            }
                                        }
                                        ?>
                                    </select>
                            </td>
                            <td>
                                    <button type="submit" class="btn btn-small"
                                            onclick="return confirm('Change this student\\'s game/nation assignment?');">
                                        Save
                                    </button>
                                </form>
                            </td>
                            <td>
                                <form method="post" class="inline-form"
                                      onsubmit="return confirm('Delete this student and free the seat?');">
                                    <input type="hidden" name="student_action" value="delete">
                                    <input type="hidden" name="student_id"
                                           value="<?php echo (int)$s['id']; ?>">
                                    <button type="submit" class="btn btn-danger btn-small">Delete</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <!-- Hidden create-game panel -->
    <div id="create-game-panel" class="card card-accent hidden">
        <?php if ($total_games >= $max_games): ?>
            <p>You have reached your game limit. Contact support to upgrade your plan.</p>
        <?php else: ?>
            <h3>Create Game</h3>
            <form method="post" class="form-grid">
                <input type="hidden" name="create_game" value="1">

                <label>
                    Game name
                    <input type="text" name="game_name" required>
                </label>

                <label>
                    Map type
                    <select name="map_type">
                        <option value="standard">Standard</option>
                        <option value="archipelago">Archipelago (many islands)</option>
                        <option value="continents">Continents</option>
                        <option value="pangea">Single supercontinent</option>
                    </select>
                </label>

                <label>
                    Map size
                    <select name="map_size">
                        <option value="small">Small</option>
                        <option value="medium" selected>Medium</option>
                        <option value="large">Large</option>
                        <option value="huge">Huge</option>
                    </select>
                </label>

                <label>
                    Number of nations
                    <input type="number" name="num_nations" value="4" min="1" max="8">
                </label>

                <input type="hidden" name="inequality_level" id="inequality-level-field" value="medium">
                <label>
                    Inequality level (wealth & population spread)
                    <div class="slider-row">
                        <input type="range" name="inequality_slider" min="0" max="1" step="0.01" value="0.5" data-preview-input>
                        <span class="value-pill" data-value-display="inequality_slider">Balanced</span>
                    </div>
                </label>

                <label>
                    Land/sea distribution
                    <select name="land_sea_pattern">
                        <option value="balanced" selected>Balanced</option>
                        <option value="land_heavy">Land-heavy</option>
                        <option value="sea_heavy">Sea-heavy</option>
                        <option value="polar_caps">Polar caps, open equator</option>
                    </select>
                    <div class="slider-row" style="margin-top:0.35rem;">
                        <input type="range" name="land_sea_ratio" min="0.25" max="0.8" step="0.01" value="0.55" data-preview-input>
                        <span class="value-pill" data-value-display="land_sea_ratio">55% land</span>
                    </div>
                </label>

                <label>
                    Preset map (optional)
                    <select name="preset_map">
                        <option value="">None (procedural)</option>
                        <option value="earth_like">Earth-like</option>
                        <option value="fragmented_world">Fragmented world</option>
                        <option value="ring_world">Ring-shaped world</option>
                    </select>
                </label>

                <label>
                    Lacunarity (map texture)
                    <div class="slider-row">
                        <input type="range" name="lacunarity" min="0.8" max="2.5" step="0.05" value="1.5" data-preview-input>
                        <span class="value-pill" data-value-display="lacunarity">1.50</span>
                    </div>
                </label>

                <label>
                    Map resolution level
                    <div class="slider-row">
                        <input type="range" name="resolution_level" min="1" max="5" step="1" value="2"
                               data-preview-input data-resolution-cap="<?php echo htmlspecialchars($resolution_cap); ?>">
                        <span class="value-pill" data-value-display="resolution_level">Standard</span>
                    </div>
                    <small class="meta">Levels above your account cap are clamped automatically.</small>
                </label>

                <div class="form-full-width">
                    <h4>Draft map preview (not yet committed)</h4>
                    <p class="meta" id="map-preview-meta">Adjust sliders to regenerate a draft map before saving turn one.</p>
                    <div id="map-preview-draft" class="map-preview muted-preview"></div>
                </div>

                <div class="form-full-width">
                    <h4>Nations configuration (optional)</h4>
                    <p class="meta">
                        Configure up to 8 nations. The first <em>Number of nations</em> rows will be used.
                        You can customize their systems and goals now, and edit flags later.
                    </p>

                    <div class="table-wrapper">
                        <table class="nation-table">
                            <thead>
                            <tr>
                                <th>#</th>
                                <th>Name</th>
                                <th>Political</th>
                                <th>Economic</th>
                                <th>Media freedom</th>
                                <th>Goals / notes</th>
                            </tr>
                            </thead>
                            <tbody>
                            <?php for ($i = 0; $i < 8; $i++): ?>
                                <tr>
                                    <td><?php echo $i + 1; ?></td>
                                    <td>
                                        <input type="text" name="nation_name[]"
                                               placeholder="Nation <?php echo $i + 1; ?>">
                                    </td>
                                    <td>
                                        <select name="nation_political_id[]">
                                            <option value="">(default)</option>
                                            <?php foreach ($political_systems as $ps): ?>
                                                <option value="<?php echo (int)$ps['id']; ?>">
                                                    <?php echo htmlspecialchars($ps['name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td>
                                        <select name="nation_economic_id[]">
                                            <option value="">(default)</option>
                                            <?php foreach ($economic_systems as $es): ?>
                                                <option value="<?php echo (int)$es['id']; ?>">
                                                    <?php echo htmlspecialchars($es['name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td>
                                        <select name="nation_media_id[]">
                                            <option value="">(default)</option>
                                            <?php foreach ($media_levels as $mf): ?>
                                                <option value="<?php echo (int)$mf['id']; ?>">
                                                    <?php echo htmlspecialchars($mf['name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td>
                                        <input type="text" name="nation_goals[]"
                                               placeholder="Key goals, themes, constraints...">
                                    </td>
                                </tr>
                            <?php endfor; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <div class="form-actions form-full-width">
                    <button type="submit" class="btn btn-primary">Create Game</button>
                    <button type="button" class="btn btn-ghost" onclick="toggleCreateGame()">
                        Cancel
                    </button>
                </div>
            </form>
        <?php endif; ?>
    </div>

    <?php if ($selected_game): ?>
        <div class="card">
            <h3>Game Details: <?php echo htmlspecialchars($selected_game['name']); ?></h3>
            <p class="meta">
                Code: <code><?php echo htmlspecialchars($selected_game['game_code']); ?></code><br>
                Map: <?php echo htmlspecialchars($selected_game['map_type']); ?> /
                     <?php echo htmlspecialchars($selected_game['map_size']); ?><br>
                Nations: <?php echo (int)$selected_game['num_nations']; ?><br>
                Inequality: <?php echo htmlspecialchars($selected_game['inequality_level']); ?><br>
                Land/sea: <?php echo htmlspecialchars($selected_game['land_sea_pattern']); ?><br>
                Preset: <?php echo htmlspecialchars($selected_game['preset_map'] ?? 'None'); ?><br>
                Current turn: <?php echo isset($selected_game['current_turn']) ? (int)$selected_game['current_turn'] : 1; ?><br>
                Turn mode: <?php echo htmlspecialchars($selected_game['turn_mode'] ?? 'manual'); ?>
            </p>

            <h4>Student team editing</h4>
            <form method="post" class="inline-form">
                <input type="hidden" name="toggle_student_edit" value="1">
                <input type="hidden" name="game_id" value="<?php echo (int)$selected_game['id']; ?>">
                <label>
                    <input type="checkbox" name="allow_student_team_edits" value="1"
                        <?php if (!empty($selected_game['allow_student_team_edits'])) echo 'checked'; ?>>
                    Allow students to edit their team assignments in-game
                </label>
                <button type="submit" class="btn btn-small">Save</button>
            </form>
        </div>

        <?php
        $map_state = [];
        if (!empty($selected_game['game_state'])) {
            $decoded = json_decode($selected_game['game_state'], true);
            if (is_array($decoded) && isset($decoded['map_options'])) {
                $map_state = $decoded['map_options'];
            }
        }

        $map_options = array_merge([
            'game_id'          => $selected_game['id'] ?? null,
            'map_type'         => $selected_game['map_type'] ?? 'standard',
            'map_size'         => $selected_game['map_size'] ?? 'medium',
            'inequality_level' => $selected_game['inequality_level'] ?? 'medium',
            'land_sea_pattern' => $selected_game['land_sea_pattern'] ?? 'balanced',
            'preset_map'       => $selected_game['preset_map'] ?? '',
            'num_nations'      => $selected_game['num_nations'] ?? 4,
            'land_sea_ratio'   => 0.5,
            'lacunarity'       => 1.5,
            'resolution_level' => $resolution_cap,
        ], $map_state);
        $selected_game_map = apw_generate_world_map($map_options);
        ?>
        <div class="card">
            <h3>Map preview</h3>
            <p class="meta">
                Uses the procedural generator (<?php echo htmlspecialchars($map_options['map_type']); ?>,
                <?php echo htmlspecialchars($map_options['map_size']); ?>) to mirror what students see.
                Resolution: <?php echo htmlspecialchars($map_options['resolution_level'] ?? 'standard'); ?> ·
                Land: <?php echo round(($map_options['land_sea_ratio'] ?? 0.5) * 100); ?>%
            </p>
            <?php echo apw_render_map_html($selected_game_map); ?>
            <p class="meta">
                Seed: <code><?php echo htmlspecialchars($selected_game_map['seed']); ?></code> ·
                Tiles: <?php echo (int)$selected_game_map['width']; ?> ×
                <?php echo (int)$selected_game_map['height']; ?>
            </p>
        </div>

        <div class="card">
            <h3>Nations in this game</h3>
            <?php if (!$selected_game_nations): ?>
                <p>No nations found (this would be unusual).</p>
            <?php else: ?>
                <div class="table-wrapper">
                    <table class="nation-table">
                        <thead>
                        <tr>
                            <th>Name</th>
                            <th>Political</th>
                            <th>Economic</th>
                            <th>Media</th>
                            <th>Flag</th>
                            <th>Colors / stripes</th>
                            <th>Goals / notes</th>
                            <th>Actions</th>
                        </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($selected_game_nations as $n): ?>
                            <?php
                            $goals_txt = '';
                            if (!empty($n['goals_json'])) {
                                $gj = json_decode($n['goals_json'], true);
                                if (is_array($gj) && isset($gj['notes'])) {
                                    $goals_txt = $gj['notes'];
                                }
                            }
                            $primary   = $n['primary_color']   ?: '#0044aa';
                            $secondary = $n['secondary_color'] ?: '';
                            $stripe    = $n['stripe_style']    ?: 'none';
                            $width_f   = $n['stripe_width_fraction'] ?: '1/2';
                            ?>
                            <tr>
                                <form method="post">
                                    <input type="hidden" name="nation_action" value="update">
                                    <input type="hidden" name="nation_id"
                                           value="<?php echo (int)$n['id']; ?>">
                                    <input type="hidden" name="game_id"
                                           value="<?php echo (int)$selected_game['id']; ?>">

                                    <td>
                                        <input type="text" name="nation_name"
                                               value="<?php echo htmlspecialchars($n['name']); ?>">
                                    </td>
                                    <td>
                                        <select name="nation_political_id">
                                            <option value="">(none)</option>
                                            <?php foreach ($political_systems as $ps): ?>
                                                <option value="<?php echo (int)$ps['id']; ?>"
                                                    <?php if ((int)$n['political_system_id'] === (int)$ps['id']) echo 'selected'; ?>>
                                                    <?php echo htmlspecialchars($ps['name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td>
                                        <select name="nation_economic_id">
                                            <option value="">(none)</option>
                                            <?php foreach ($economic_systems as $es): ?>
                                                <option value="<?php echo (int)$es['id']; ?>"
                                                    <?php if ((int)$n['economic_system_id'] === (int)$es['id']) echo 'selected'; ?>>
                                                    <?php echo htmlspecialchars($es['name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td>
                                        <select name="nation_media_id">
                                            <option value="">(none)</option>
                                            <?php foreach ($media_levels as $mf): ?>
                                                <option value="<?php echo (int)$mf['id']; ?>"
                                                    <?php if ((int)$n['media_freedom_id'] === (int)$mf['id']) echo 'selected'; ?>>
                                                    <?php echo htmlspecialchars($mf['name']); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td class="flag-cell">
                                        <div class="flag-preview-small flag-preview-stripes"
                                             data-primary="<?php echo htmlspecialchars($primary); ?>"
                                             data-secondary="<?php echo htmlspecialchars($secondary); ?>"
                                             data-stripe="<?php echo htmlspecialchars($stripe); ?>"
                                             data-fraction="<?php echo htmlspecialchars($width_f); ?>"
                                             style="background: <?php echo htmlspecialchars($primary); ?>;">
                                        </div>
                                    </td>
                                    <td>
                                        <div style="display:flex;flex-direction:column;gap:0.2rem;">
                                            <label>
                                                Primary
                                                <input type="color" name="nation_primary_color"
                                                       value="<?php echo htmlspecialchars($primary); ?>">
                                            </label>
                                            <label>
                                                Secondary
                                                <input type="color" name="nation_secondary_color"
                                                       value="<?php echo htmlspecialchars($secondary ?: '#ffffff'); ?>">
                                                <small>(leave blank to remove)</small>
                                            </label>
                                            <label>
                                                Stripe style
                                                <select name="nation_stripe_style">
                                                    <option value="none" <?php if ($stripe === 'none') echo 'selected'; ?>>None</option>
                                                    <option value="horizontal" <?php if ($stripe === 'horizontal') echo 'selected'; ?>>Horizontal</option>
                                                    <option value="vertical" <?php if ($stripe === 'vertical') echo 'selected'; ?>>Vertical</option>
                                                    <option value="diagonal" <?php if ($stripe === 'diagonal') echo 'selected'; ?>>Diagonal</option>
                                                </select>
                                            </label>
                                            <label>
                                                Stripe width
                                                <select name="nation_stripe_width_fraction">
                                                    <?php
                                                    $fractions = ['1/2','1/3','1/4','1/5','1/6','1/7','1/8'];
                                                    foreach ($fractions as $f) {
                                                        $sel = ($width_f === $f) ? 'selected' : '';
                                                        echo "<option value=\"$f\" $sel>$f</option>";
                                                    }
                                                    ?>
                                                </select>
                                            </label>
                                        </div>
                                    </td>
                                    <td>
                                        <input type="text" name="nation_goals"
                                               value="<?php echo htmlspecialchars($goals_txt); ?>">
                                    </td>
                                    <td>
                                        <div class="inline-form">
                                            <button type="submit" class="btn btn-small"
                                                    onclick="return confirm('Save changes to this nation?');">
                                                Save
                                            </button>
                                        </div>
                                    </td>
                                </form>
                                <td>
                                    <form method="post" class="inline-form"
                                          onsubmit="return confirm('Delete this nation? Students assigned to it will have no nation.');">
                                        <input type="hidden" name="nation_action" value="delete">
                                        <input type="hidden" name="nation_id"
                                               value="<?php echo (int)$n['id']; ?>">
                                        <input type="hidden" name="game_id"
                                               value="<?php echo (int)$selected_game['id']; ?>">
                                        <button type="submit" class="btn btn-danger btn-small">Delete</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <h4>Add nation (only before turn 2)</h4>
                <form method="post" class="form-grid">
                    <input type="hidden" name="nation_action" value="create">
                    <input type="hidden" name="game_id"
                           value="<?php echo (int)$selected_game['id']; ?>">

                    <label>
                        Name
                        <input type="text" name="nation_name" placeholder="Optional – auto named if blank">
                    </label>
                    <label>
                        Political
                        <select name="nation_political_id">
                            <option value="">(none)</option>
                            <?php foreach ($political_systems as $ps): ?>
                                <option value="<?php echo (int)$ps['id']; ?>">
                                    <?php echo htmlspecialchars($ps['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </label>
                    <label>
                        Economic
                        <select name="nation_economic_id">
                            <option value="">(none)</option>
                            <?php foreach ($economic_systems as $es): ?>
                                <option value="<?php echo (int)$es['id']; ?>">
                                    <?php echo htmlspecialchars($es['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </label>
                    <label>
                        Media freedom
                        <select name="nation_media_id">
                            <option value="">(none)</option>
                            <?php foreach ($media_levels as $mf): ?>
                                <option value="<?php echo (int)$mf['id']; ?>">
                                    <?php echo htmlspecialchars($mf['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </label>

                    <label>
                        Primary color
                        <input type="color" name="nation_primary_color" value="#0044aa">
                    </label>
                    <label>
                        Secondary color
                        <input type="color" name="nation_secondary_color" value="#ffffff">
                        <small>(leave blank to remove)</small>
                    </label>
                    <label>
                        Stripe style
                        <select name="nation_stripe_style">
                            <option value="none">None</option>
                            <option value="horizontal">Horizontal</option>
                            <option value="vertical">Vertical</option>
                            <option value="diagonal">Diagonal</option>
                        </select>
                    </label>
                    <label>
                        Stripe width
                        <select name="nation_stripe_width_fraction">
                            <option value="1/2">1/2</option>
                            <option value="1/3">1/3</option>
                            <option value="1/4" selected>1/4</option>
                            <option value="1/5">1/5</option>
                            <option value="1/6">1/6</option>
                            <option value="1/7">1/7</option>
                            <option value="1/8">1/8</option>
                        </select>
                    </label>

                    <label class="form-full-width">
                        Goals / notes
                        <input type="text" name="nation_goals"
                               placeholder="Key goals, narratives, constraints...">
                    </label>

                    <div class="form-actions form-full-width">
                        <button type="submit" class="btn btn-primary">Add Nation</button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <script>
    (function(){
        const form = document.querySelector('#create-game-panel form');
        if (!form) return;

        const previewTarget = document.getElementById('map-preview-draft');
        const previewMeta = document.getElementById('map-preview-meta');
        const inequalityField = document.getElementById('inequality-level-field');
        const controls = form.querySelectorAll('[data-preview-input]');

        const resolutionMap = ['starter','standard','advanced','district','enterprise'];
        const resolutionCap = form.querySelector('input[name="resolution_level"]').dataset.resolutionCap;

        const clampResolution = (val) => {
            const capIndex = Math.max(0, resolutionMap.indexOf(resolutionCap || 'standard'));
            return Math.min(val, capIndex + 1);
        };

        const displayValue = (input) => {
            const pill = form.querySelector(`[data-value-display="${input.name}"]`);
            if (!pill) return;
            let text = input.value;
            if (input.name === 'land_sea_ratio') {
                text = Math.round(parseFloat(input.value) * 100) + '% land';
            } else if (input.name === 'lacunarity') {
                text = parseFloat(input.value).toFixed(2);
            } else if (input.name === 'resolution_level') {
                const idx = Math.max(0, Math.min(resolutionMap.length - 1, parseInt(input.value, 10) - 1));
                text = resolutionMap[idx].charAt(0).toUpperCase() + resolutionMap[idx].slice(1);
            } else if (input.name === 'inequality_slider') {
                const val = parseFloat(input.value);
                if (val < 0.34) text = 'Low inequality';
                else if (val > 0.67) text = 'High inequality';
                else text = 'Balanced';
            }
            pill.textContent = text;
        };

        const syncInequality = () => {
            const slider = form.querySelector('input[name="inequality_slider"]');
            if (!slider || !inequalityField) return;
            const val = parseFloat(slider.value);
            const level = val < 0.34 ? 'low' : (val > 0.67 ? 'high' : 'medium');
            inequalityField.value = level;
        };

        const syncResolutionCap = () => {
            const resInput = form.querySelector('input[name="resolution_level"]');
            if (!resInput) return;
            const clamped = clampResolution(parseInt(resInput.value, 10));
            resInput.value = clamped;
            displayValue(resInput);
        };

        let previewTimer;
        const schedulePreview = () => {
            if (previewTimer) clearTimeout(previewTimer);
            previewTimer = setTimeout(runPreview, 250);
        };

        const runPreview = async () => {
            syncResolutionCap();
            syncInequality();
            const fd = new FormData();
            fd.append('map_type', form.map_type.value);
            fd.append('map_size', form.map_size.value);
            fd.append('num_nations', form.num_nations.value || 4);
            fd.append('land_sea_pattern', form.land_sea_pattern.value);
            fd.append('preset_map', form.preset_map.value);
            fd.append('land_sea_ratio', form.land_sea_ratio.value);
            fd.append('lacunarity', form.lacunarity.value);
            fd.append('resolution_level', form.resolution_level.value);
            fd.append('inequality_slider', form.inequality_slider.value);

            previewMeta.textContent = 'Generating preview...';
            try {
                const resp = await fetch('map_generate_preview.php', {method: 'POST', body: fd});
                const data = await resp.json();
                if (data.html) {
                    previewTarget.innerHTML = data.html;
                    previewMeta.textContent = data.summary || 'Preview ready.';
                } else {
                    previewTarget.innerHTML = '<p>Preview unavailable.</p>';
                    previewMeta.textContent = 'Could not render preview.';
                }
            } catch (e) {
                previewTarget.innerHTML = '<p>Preview unavailable.</p>';
                previewMeta.textContent = 'Network error while generating preview.';
            }
        };

        controls.forEach((ctrl) => {
            displayValue(ctrl);
            ctrl.addEventListener('input', () => {
                if (ctrl.name === 'resolution_level') {
                    syncResolutionCap();
                }
                if (ctrl.name === 'inequality_slider') {
                    syncInequality();
                }
                displayValue(ctrl);
                schedulePreview();
            });
        });

        form.map_type.addEventListener('change', schedulePreview);
        form.map_size.addEventListener('change', schedulePreview);
        form.num_nations.addEventListener('change', schedulePreview);
        form.preset_map.addEventListener('change', schedulePreview);
        form.land_sea_pattern.addEventListener('change', schedulePreview);

        syncResolutionCap();
        syncInequality();
        runPreview();
    })();
    </script>
</section>
