<?php
// gameapp/teacher_dashboard.php
require __DIR__ . '/auth.php';

$user = require_login();
$pdo  = get_pdo();

/**
 * Load teacher + account type
 */
$stmt = $pdo->prepare(
    'SELECT u.id, u.email, u.max_students, u.account_type_id,
            at.name AS account_name, at.max_students AS at_max_students,
            at.max_games AS at_max_games, at.description AS account_description
     FROM users u
     LEFT JOIN account_types at ON u.account_type_id = at.id
     WHERE u.id = ?'
);
$stmt->execute([$user['id']]);
$teacher = $stmt->fetch();

$account_name        = $teacher['account_name'] ?? 'Custom / Unknown';
$account_description = $teacher['account_description'] ?? 'Contact support for details.';

// Seat limit: prefer account type max; fallback to user.max_students; else 3
$max_students = isset($teacher['at_max_students'])
    ? (int)$teacher['at_max_students']
    : (int)($teacher['max_students'] ?? 3);

// Game limit: prefer account type; else high number
$max_games = isset($teacher['at_max_games'])
    ? (int)$teacher['at_max_games']
    : 999;

/**
 * Load political, economic, and media freedom systems
 */
$stmt = $pdo->query('SELECT id, name FROM political_systems ORDER BY name ASC');
$political_systems = $stmt->fetchAll();

$stmt = $pdo->query('SELECT id, name FROM economic_systems ORDER BY name ASC');
$economic_systems = $stmt->fetchAll();

$stmt = $pdo->query('SELECT id, name FROM media_freedom ORDER BY id ASC');
$media_levels = $stmt->fetchAll();

/**
 * helper: generate code
 */
if (!function_exists('generate_code')) {
    function generate_code($len = 8): string {
        $chars = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
        $out = '';
        for ($i = 0; $i < $len; $i++) {
            $out .= $chars[random_int(0, strlen($chars) - 1)];
        }
        return $out;
    }
}

/**
 * Handle Create Student (global, up to seat limit)
 */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_student'])) {
    $stmt = $pdo->prepare('SELECT COUNT(*) AS cnt FROM students WHERE teacher_id = ?');
    $stmt->execute([$teacher['id']]);
    $row  = $stmt->fetch();
    $used = (int)$row['cnt'];

    if ($used < $max_students) {
        $label = trim($_POST['label'] ?? '');
        $student_code = generate_code(10);

        $stmt = $pdo->prepare(
            'INSERT INTO students (teacher_id, student_code, label, game_id)
             VALUES (?, ?, ?, NULL)'
        );
        $stmt->execute([$teacher['id'], $student_code, $label ?: null]);
    }
    header('Location: teacher_dashboard.php?tab=games');
    exit;
}

/**
 * Move / delete student, including game+nation assignment
 */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['student_action'])) {
    $student_id = (int)($_POST['student_id'] ?? 0);

    $stmt = $pdo->prepare('SELECT * FROM students WHERE id = ? AND teacher_id = ?');
    $stmt->execute([$student_id, $teacher['id']]);
    $student = $stmt->fetch();

    if ($student) {
        if ($_POST['student_action'] === 'delete') {
            $stmt = $pdo->prepare('DELETE FROM students WHERE id = ?');
            $stmt->execute([$student_id]);

        } elseif ($_POST['student_action'] === 'assign_game_nation') {
            $new_game_id   = (int)($_POST['student_game']   ?? 0);
            $new_nation_id = (int)($_POST['student_nation'] ?? 0);

            if ($new_game_id > 0) {
                $stmt = $pdo->prepare('SELECT id FROM games WHERE id = ? AND teacher_id = ?');
                $stmt->execute([$new_game_id, $teacher['id']]);
                $game = $stmt->fetch();

                if ($game) {
                    // Set student's game
                    $stmt = $pdo->prepare('UPDATE students SET game_id = ? WHERE id = ?');
                    $stmt->execute([$new_game_id, $student_id]);

                    // Clear existing nation membership for that game
                    $stmt = $pdo->prepare(
                        'DELETE nm FROM nation_members nm
                         JOIN nations n ON nm.nation_id = n.id
                         WHERE nm.student_id = ? AND n.game_id = ?'
                    );
                    $stmt->execute([$student_id, $new_game_id]);

                    // Assign to nation if chosen
                    if ($new_nation_id > 0) {
                        $stmt = $pdo->prepare(
                            'INSERT INTO nation_members (nation_id, student_id, role_label, privileges_json)
                             VALUES (?, ?, NULL, NULL)'
                        );
                        $stmt->execute([$new_nation_id, $student_id]);
                    }
                }
            } else {
                // No game: clear memberships and game_id
                $stmt = $pdo->prepare('DELETE FROM nation_members WHERE student_id = ?');
                $stmt->execute([$student_id]);

                $stmt = $pdo->prepare('UPDATE students SET game_id = NULL WHERE id = ?');
                $stmt->execute([$student_id]);
            }
        }
    }

    header('Location: teacher_dashboard.php?tab=games');
    exit;
}

/**
 * Handle Create Game (with options + per-nation config)
 */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_game'])) {
    $stmt = $pdo->prepare('SELECT COUNT(*) AS cnt FROM games WHERE teacher_id = ?');
    $stmt->execute([$teacher['id']]);
    $row  = $stmt->fetch();
    $used_games = (int)$row['cnt'];

    if ($used_games < $max_games) {
        $name            = trim($_POST['game_name'] ?? '');
        $map_type        = trim($_POST['map_type'] ?? 'standard');
        $map_size        = trim($_POST['map_size'] ?? 'medium');
        $num_nations     = max(1, (int)($_POST['num_nations'] ?? 4));
        $inequality_raw  = isset($_POST['inequality_slider']) ? (float)$_POST['inequality_slider'] : 0.5;
        $inequality      = ($inequality_raw < 0.34) ? 'low' : (($inequality_raw > 0.67) ? 'high' : 'medium');
        $land_sea        = trim($_POST['land_sea_pattern'] ?? 'balanced');
        $preset_map      = trim($_POST['preset_map'] ?? '');
        $land_sea_ratio  = isset($_POST['land_sea_ratio']) ? max(0.1, min(0.9, (float)$_POST['land_sea_ratio'])) : 0.5;
        $lacunarity      = isset($_POST['lacunarity']) ? max(0.6, min(3.0, (float)$_POST['lacunarity'])) : 1.5;
        $resolution_lvl  = trim($_POST['resolution_level'] ?? 'standard');

        $nation_names       = $_POST['nation_name']         ?? [];
        $nation_pol_ids     = $_POST['nation_political_id'] ?? [];
        $nation_ec_ids      = $_POST['nation_economic_id']  ?? [];
        $nation_media_ids   = $_POST['nation_media_id']     ?? [];
        $nation_goals       = $_POST['nation_goals']        ?? [];

        $game_state = [
            'map_options' => [
                'map_type' => $map_type,
                'map_size' => $map_size,
                'num_nations' => $num_nations,
                'land_sea_pattern' => $land_sea,
                'preset_map' => $preset_map,
                'land_sea_ratio' => $land_sea_ratio,
                'lacunarity' => $lacunarity,
                'resolution_level' => $resolution_lvl,
                'inequality_slider' => $inequality_raw,
                'inequality_level' => $inequality,
            ],
        ];

        if ($name !== '') {
            $game_code = generate_code(8);

            $stmt = $pdo->prepare(
                'INSERT INTO games (
                    teacher_id, name, game_code, game_state,
                    map_type, map_size, num_nations,
                    inequality_level, land_sea_pattern, preset_map,
                    allow_student_team_edits,
                    current_turn, turn_mode, turn_interval_minutes,
                    scheduled_daily_time, next_turn_at, last_turn_at
                 ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 0, 1, "manual", NULL, NULL, NULL, NULL)'
            );

            // NOTE: minimal turn fields; adjust as needed if you want real timings
            $stmt->execute([
                $teacher['id'],
                $name,
                $game_code,
                json_encode($game_state),
                $map_type,
                $map_size,
                $num_nations,
                $inequality,
                $land_sea,
                $preset_map ?: null
            ]);

            $game_id = (int)$pdo->lastInsertId();

            // Create up to num_nations with default flags
            for ($i = 0; $i < $num_nations; $i++) {
                $raw_name = trim($nation_names[$i] ?? '');
                $n_name   = $raw_name !== '' ? $raw_name : ('Nation ' . ($i + 1));

                $pol_id   = !empty($nation_pol_ids[$i])   ? (int)$nation_pol_ids[$i]   : null;
                $ec_id    = !empty($nation_ec_ids[$i])    ? (int)$nation_ec_ids[$i]    : null;
                $media_id = !empty($nation_media_ids[$i]) ? (int)$nation_media_ids[$i] : null;

                $goals_text = trim($nation_goals[$i] ?? '');
                $goals_json = $goals_text !== '' ? json_encode(['notes' => $goals_text]) : null;

                $stmt = $pdo->prepare(
                    'INSERT INTO nations (
                        game_id, name,
                        political_system_id,
                        economic_system_id,
                        media_freedom_id,
                        goals_json,
                        primary_color,
                        secondary_color,
                        stripe_style,
                        stripe_width_fraction
                     ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)'
                );
                $stmt->execute([
                    $game_id,
                    $n_name,
                    $pol_id,
                    $ec_id,
                    $media_id,
                    $goals_json,
                    '#0044aa',      // default primary
                    null,           // secondary
                    'none',         // no stripes
                    '1/2'
                ]);
            }
        }
    }

    header('Location: teacher_dashboard.php?tab=games');
    exit;
}

/**
 * Handle game deletion (only if no turns exist)
 */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_game'])) {
    $game_id = (int)($_POST['game_id'] ?? 0);

    // Ensure game belongs to teacher
    $stmt = $pdo->prepare('SELECT id FROM games WHERE id = ? AND teacher_id = ?');
    $stmt->execute([$game_id, $teacher['id']]);
    $game = $stmt->fetch();

    if ($game) {
        $stmt = $pdo->prepare('SELECT COUNT(*) AS cnt FROM game_turns WHERE game_id = ?');
        $stmt->execute([$game_id]);
        $row = $stmt->fetch();
        $turns = (int)$row['cnt'];

        if ($turns === 0) {
            $stmt = $pdo->prepare('DELETE FROM games WHERE id = ?');
            $stmt->execute([$game_id]);
            header('Location: teacher_dashboard.php?tab=games');
            exit;
        } else {
            header('Location: teacher_dashboard.php?tab=games&delete_error=1');
            exit;
        }
    }

    header('Location: teacher_dashboard.php?tab=games');
    exit;
}

/**
 * Handle toggle: allow_student_team_edits
 */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['toggle_student_edit'])) {
    $game_id = (int)($_POST['game_id'] ?? 0);

    $stmt = $pdo->prepare('SELECT id FROM games WHERE id = ? AND teacher_id = ?');
    $stmt->execute([$game_id, $teacher['id']]);
    $game = $stmt->fetch();

    if ($game) {
        $value = isset($_POST['allow_student_team_edits']) ? 1 : 0;
        $stmt = $pdo->prepare(
            'UPDATE games SET allow_student_team_edits = ? WHERE id = ?'
        );
        $stmt->execute([$value, $game_id]);
    }

    header('Location: teacher_dashboard.php?tab=games&game=' . $game_id);
    exit;
}

/**
 * Handle nation editing (create / update / delete)
 * Nations cannot be added after turn 1 has been played.
 */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['nation_action'])) {
    $action = $_POST['nation_action'];

    if ($action === 'update') {
        $nation_id = (int)($_POST['nation_id'] ?? 0);
        $game_id   = (int)($_POST['game_id'] ?? 0);

        $stmt = $pdo->prepare(
            'SELECT n.id, n.game_id, g.current_turn
             FROM nations n
             JOIN games g ON n.game_id = g.id
             WHERE n.id = ? AND g.teacher_id = ?'
        );
        $stmt->execute([$nation_id, $teacher['id']]);
        $nrow = $stmt->fetch();

        if ($nrow) {
            $name      = trim($_POST['nation_name'] ?? '');
            $pol_id    = ($_POST['nation_political_id'] === '' ? null : (int)$_POST['nation_political_id']);
            $ec_id     = ($_POST['nation_economic_id'] === ''  ? null : (int)$_POST['nation_economic_id']);
            $media_id  = ($_POST['nation_media_id'] === ''     ? null : (int)$_POST['nation_media_id']);
            $goals_txt = trim($_POST['nation_goals'] ?? '');
            $goals_json = $goals_txt !== '' ? json_encode(['notes' => $goals_txt]) : null;

            $primary   = trim($_POST['nation_primary_color'] ?? '#0044aa');
            if ($primary === '') $primary = '#0044aa';
            $secondary = trim($_POST['nation_secondary_color'] ?? '');
            if ($secondary === '') $secondary = null;
            $stripe    = $_POST['nation_stripe_style'] ?? 'none';
            if (!$secondary) {
                $stripe = 'none';
            }
            $stripe_width = $_POST['nation_stripe_width_fraction'] ?? '1/2';

            if ($name === '') {
                $name = 'Nation ' . $nation_id;
            }

            $stmt = $pdo->prepare(
                'UPDATE nations
                 SET name = ?, political_system_id = ?, economic_system_id = ?,
                     media_freedom_id = ?, goals_json = ?,
                     primary_color = ?, secondary_color = ?, stripe_style = ?, stripe_width_fraction = ?
                 WHERE id = ?'
            );
            $stmt->execute([
                $name,
                $pol_id,
                $ec_id,
                $media_id,
                $goals_json,
                $primary,
                $secondary,
                $stripe,
                $stripe_width,
                $nation_id
            ]);
        }

        header('Location: teacher_dashboard.php?tab=games&game=' . $game_id);
        exit;
    }

    if ($action === 'create') {
        $game_id   = (int)($_POST['game_id'] ?? 0);

        $stmt = $pdo->prepare(
            'SELECT id, current_turn FROM games WHERE id = ? AND teacher_id = ?'
        );
        $stmt->execute([$game_id, $teacher['id']]);
        $game = $stmt->fetch();

        // disallow creation if game has advanced beyond turn 1
        if ($game && (int)$game['current_turn'] <= 1) {
            $name      = trim($_POST['nation_name'] ?? '');
            $pol_id    = ($_POST['nation_political_id'] === '' ? null : (int)$_POST['nation_political_id']);
            $ec_id     = ($_POST['nation_economic_id'] === ''  ? null : (int)$_POST['nation_economic_id']);
            $media_id  = ($_POST['nation_media_id'] === ''     ? null : (int)$_POST['nation_media_id']);
            $goals_txt = trim($_POST['nation_goals'] ?? '');
            $goals_json = $goals_txt !== '' ? json_encode(['notes' => $goals_txt]) : null;

            $primary   = trim($_POST['nation_primary_color'] ?? '#0044aa');
            if ($primary === '') $primary = '#0044aa';
            $secondary = trim($_POST['nation_secondary_color'] ?? '');
            if ($secondary === '') $secondary = null;
            $stripe    = $_POST['nation_stripe_style'] ?? 'none';
            if (!$secondary) {
                $stripe = 'none';
            }
            $stripe_width = $_POST['nation_stripe_width_fraction'] ?? '1/2';

            if ($name === '') {
                $stmt = $pdo->prepare('SELECT COUNT(*) AS cnt FROM nations WHERE game_id = ?');
                $stmt->execute([$game_id]);
                $row = $stmt->fetch();
                $next = (int)$row['cnt'] + 1;
                $name = 'Nation ' . $next;
            }

            $stmt = $pdo->prepare(
                'INSERT INTO nations (
                    game_id, name,
                    political_system_id, economic_system_id,
                    media_freedom_id, goals_json,
                    primary_color, secondary_color,
                    stripe_style, stripe_width_fraction
                 ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)'
            );
            $stmt->execute([
                $game_id,
                $name,
                $pol_id,
                $ec_id,
                $media_id,
                $goals_json,
                $primary,
                $secondary,
                $stripe,
                $stripe_width
            ]);
        }

        header('Location: teacher_dashboard.php?tab=games&game=' . $game_id);
        exit;
    }

    if ($action === 'delete') {
        $nation_id = (int)($_POST['nation_id'] ?? 0);
        $game_id   = (int)($_POST['game_id'] ?? 0);

        $stmt = $pdo->prepare(
            'SELECT n.id
             FROM nations n
             JOIN games g ON n.game_id = g.id
             WHERE n.id = ? AND g.teacher_id = ?'
        );
        $stmt->execute([$nation_id, $teacher['id']]);
        $nrow = $stmt->fetch();

        if ($nrow) {
            $stmt = $pdo->prepare('DELETE FROM nations WHERE id = ?');
            $stmt->execute([$nation_id]);
        }

        header('Location: teacher_dashboard.php?tab=games&game=' . $game_id);
        exit;
    }
}

/**
 * Load games
 */
$stmt = $pdo->prepare('SELECT * FROM games WHERE teacher_id = ? ORDER BY created_at DESC');
$stmt->execute([$teacher['id']]);
$games = $stmt->fetchAll();
$total_games = count($games);

/**
 * Load students (+ game & nation display)
 */
$stmt = $pdo->prepare(
    'SELECT s.*,
            g.name AS game_name,
            n.name AS nation_name,
            n.id   AS nation_id
     FROM students s
     LEFT JOIN games g ON s.game_id = g.id
     LEFT JOIN nation_members nm ON nm.student_id = s.id
     LEFT JOIN nations n ON nm.nation_id = n.id
     WHERE s.teacher_id = ?
     GROUP BY s.id
     ORDER BY s.created_at ASC'
);
$stmt->execute([$teacher['id']]);
$students = $stmt->fetchAll();

$used_seats  = count($students);
$empty_seats = max(0, $max_students - $used_seats);

/**
 * Build game -> nations map (with system & media names + flags)
 */
$game_nations = [];
$game_nations_for_js = [];
if ($games) {
    $game_ids = array_column($games, 'id');
    $placeholders = implode(',', array_fill(0, count($game_ids), '?'));

    $stmt = $pdo->prepare(
        "SELECT n.*,
                ps.name AS political_name,
                es.name AS economic_name,
                mf.name AS media_name
         FROM nations n
         LEFT JOIN political_systems ps ON n.political_system_id = ps.id
         LEFT JOIN economic_systems es ON n.economic_system_id = es.id
         LEFT JOIN media_freedom mf ON n.media_freedom_id = mf.id
         WHERE n.game_id IN ($placeholders)
         ORDER BY n.game_id, n.id ASC"
    );
    $stmt->execute($game_ids);
    $rows = $stmt->fetchAll();

    foreach ($rows as $row) {
        $game_nations[$row['game_id']][] = $row;
        $game_nations_for_js[$row['game_id']][] = [
            'id'   => (int)$row['id'],
            'name' => $row['name'],
        ];
    }
}

/**
 * Selected game (for detail in Games tab)
 */
$selected_game = null;
$selected_game_nations = [];
if (isset($_GET['game'])) {
    $gid = (int)$_GET['game'];
    foreach ($games as $g) {
        if ($g['id'] == $gid) {
            $selected_game = $g;
            break;
        }
    }
    if ($selected_game) {
        $selected_game_nations = $game_nations[$selected_game['id']] ?? [];
    }
}

/**
 * Active tab and delete-error flag
 */
$tab = $_GET['tab'] ?? 'games';
$valid_tabs = ['games', 'students', 'account'];
if (!in_array($tab, $valid_tabs, true)) {
    $tab = 'games';
}
$delete_error = isset($_GET['delete_error']);
?>
<!doctype html>
<html>
<head>
    <meta charset="utf-8">
    <title>Teacher Dashboard – A Possible World</title>
    <link rel="stylesheet" href="style.css">
    <style>
        .flag-preview-small {
            width: 60px;
            height: 36px;
            border-radius: 4px;
            border: 1px solid rgba(0,0,0,0.3);
            overflow: hidden;
        }
        .nation-table th,
        .nation-table td {
            padding: 0.2rem 0.3rem;
            font-size: 0.85rem;
        }
        .nation-table input[type="text"] {
            width: 110px;
        }
        .nation-table select {
            max-width: 110px;
        }
        .nation-table .flag-cell {
            width: 80px;
        }
    </style>
    <script>
        const NATIONS_BY_GAME = <?php echo json_encode($game_nations_for_js, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_AMP | JSON_HEX_QUOT); ?>;

        function switchTab(tabName) {
            const panes = document.querySelectorAll('.tab-pane');
            const buttons = document.querySelectorAll('.tabs button');
            panes.forEach(p => p.classList.remove('active'));
            buttons.forEach(b => b.classList.remove('active'));

            const pane = document.getElementById('tab-' + tabName);
            const btn  = document.getElementById('btn-' + tabName);
            if (pane) pane.classList.add('active');
            if (btn)  btn.classList.add('active');

            const url = new URL(window.location.href);
            url.searchParams.set('tab', tabName);
            if (tabName !== 'games') {
                url.searchParams.delete('game');
                url.searchParams.delete('delete_error');
            }
            window.history.replaceState({}, '', url);
        }

        function toggleCreateGame() {
            const panel = document.getElementById('create-game-panel');
            if (!panel) return;
            panel.classList.toggle('hidden');
        }

        function setupStudentGameNationDropdowns() {
            document.querySelectorAll('.student-game-select').forEach(select => {
                select.addEventListener('change', function () {
                    const gameId = parseInt(this.value, 10) || 0;
                    const row = this.closest('tr');
                    const nationSelect = row.querySelector('.student-nation-select');
                    if (!nationSelect) return;

                    while (nationSelect.firstChild) {
                        nationSelect.removeChild(nationSelect.firstChild);
                    }
                    const optNone = new Option('No nation', '0');
                    nationSelect.add(optNone);

                    if (gameId > 0 && NATIONS_BY_GAME[gameId]) {
                        NATIONS_BY_GAME[gameId].forEach(n => {
                            const opt = new Option(n.name, n.id);
                            nationSelect.add(opt);
                        });
                    }
                });
            });
        }

        window.addEventListener('DOMContentLoaded', function () {
            switchTab('<?php echo htmlspecialchars($tab, ENT_QUOTES); ?>');
            setupStudentGameNationDropdowns();
        });
    </script>
</head>
<body class="app-body">
<div class="app-shell">
    <header class="app-header">
        <div>
            <div class="app-title">A Possible World – Teacher Dashboard</div>
            <div class="app-subtitle">Logged in as <?php echo htmlspecialchars($teacher['email']); ?></div>
        </div>
        <div>
            <a class="btn btn-ghost" href="logout.php">Logout</a>
        </div>
    </header>

    <main class="app-main">
        <div class="tabs">
            <button id="btn-games" onclick="switchTab('games')">Games</button>
            <button id="btn-students" onclick="switchTab('students')">Students</button>
            <button id="btn-account" onclick="switchTab('account')">Account</button>
        </div>

        <?php include __DIR__ . '/teacher_tab_students.php'; ?>
        <?php include __DIR__ . '/teacher_tab_games.php'; ?>
        <?php include __DIR__ . '/teacher_tab_account.php'; ?>
    </main>
</div>
</body>
</html>
