<?php
// gameapp/sim_engine.php
// Deterministic per-turn engine implementing the core “A Possible World” rules:
// - tile-level food/energy/industry/environment/population updates
// - national aggregation, policies, corruption/media/education/SoL/happiness
// - military recruitment/readiness/equipment/cost
// Constants live in apw_default_engine_config() for easy tuning.

if (!function_exists('apw_clamp')) {
    function apw_clamp(float $v, float $min, float $max): float {
        if ($v < $min) {
            return $min;
        }
        if ($v > $max) {
            return $max;
        }
        return $v;
    }
}

if (!function_exists('apw_safe_div')) {
    function apw_safe_div(float $num, float $den, float $default = 0.0): float {
        return ($den !== 0.0) ? ($num / $den) : $default;
    }
}

if (!function_exists('apw_default_engine_config')) {
    /**
     * Tunable constants for all formulas. Values are moderate so the sim stays stable.
     */
    function apw_default_engine_config(): array {
        return [
            'base_budget' => 100.0,
            'budget_lambda_industry' => 0.02,
            'eps' => 1e-6,
            'k_dens' => 10.0,
            'tech_max' => 10.0,
            'investment_loss_from_corruption' => 0.6, // lambda_c
            'tile_constants' => [
                'food' => ['a1' => 1.0, 'alpha1' => 0.5, 'alpha2' => 0.05, 'eta1' => 0.05],
                'energy' => [
                    'b1' => 0.5,
                    'b2' => 0.3,
                    'b3' => 0.2,
                    'beta1' => 0.4,
                    'beta2' => 0.6,
                    'beta3' => 0.4,
                    'eta2' => 0.05,
                    'eta3' => 0.05,
                ],
                'infra' => ['gamma1' => 0.35, 'gamma2' => 0.04, 'eta4' => 0.05],
                'industry' => [
                    'delta_g' => 0.4,
                    'delta_s' => 0.35,
                    'delta_h' => 0.5,
                    'eta_g' => 0.06,
                    'eta_s' => 0.05,
                    'eta_h' => 0.06,
                    'penalty_pollution_goods' => 0.02,
                    'penalty_pollution_services' => 0.02,
                    'penalty_pollution_heavy' => 0.025,
                ],
                'env' => [
                    'phi1' => 0.15,
                    'phi2' => 0.08,
                    'phi3' => 0.3,
                    'psi1' => 0.2,
                    'psi2' => 0.08,
                    'eta_env' => 0.05,
                ],
                'population' => [
                    'w1' => 0.4,
                    'w2' => 0.2,
                    'w3' => 0.2,
                    'w4' => 0.2,
                    'rho1' => 0.08,
                    'rho2' => 0.04,
                    'fsr_cap' => 1.5,
                ],
                'resources' => [
                    'kappa1' => 0.4,
                    'kappa2' => 0.5,
                    'lambda1' => 0.05, // forestry hurts env
                    'lambda2' => 0.06, // mining adds pollution
                    'eta_f' => 0.05,
                    'eta_m' => 0.05,
                ],
            ],
            'biome_factors' => [
                'forest' => ['forest' => 1.0, 'mineral' => 0.4],
                'plains' => ['forest' => 0.4, 'mineral' => 0.3],
                'wetland' => ['forest' => 0.8, 'mineral' => 0.2],
                'desert' => ['forest' => 0.05, 'mineral' => 0.25],
                'hill' => ['forest' => 0.2, 'mineral' => 0.8],
                'mountain' => ['forest' => 0.15, 'mineral' => 1.0],
                'coastal' => ['forest' => 0.3, 'mineral' => 0.35],
                'urban' => ['forest' => 0.1, 'mineral' => 0.25],
                'tundra' => ['forest' => 0.15, 'mineral' => 0.3],
                'water' => ['forest' => 0.0, 'mineral' => 0.0],
            ],
            'econ_balance' => ['chi1' => 0.6, 'chi2' => 0.6, 'chi3' => 0.4, 'chi4' => 0.3],
            'happiness' => ['h1' => 0.25, 'h2' => 0.2, 'h3' => 0.2, 'h4' => 0.2, 'h5' => 0.15],
            'tech_growth' => [
                'base_research' => 0.02,
                'sigma2' => 0.5,
                'tau' => [
                    'agriculture' => 0.08,
                    'industry' => 0.08,
                    'energy' => 0.08,
                    'infrastructure' => 0.08,
                    'medicine' => 0.05,
                    'military_ground' => 0.05,
                    'military_air' => 0.05,
                    'military_naval' => 0.05,
                ],
            ],
            'education' => [
                'theta1' => 1.2,
                'sigma1' => 0.5,
                'theta2' => 30.0,
                'pop_strain_threshold' => 0.04,
            ],
            'corruption' => [
                'c1' => 10.0,
                'c2' => 30.0,
                'c3' => 20.0,
                'c4' => 15.0,
                'bias' => ['democracy' => -0.5, 'social_democracy' => -1.0, 'authoritarian' => 0.5, 'oligarchy' => 1.0, 'theocracy' => 0.3, 'military_junta' => 0.8],
            ],
            'media' => [
                'omega' => 0.25,
                'caps' => ['democracy' => 100, 'social_democracy' => 100, 'authoritarian' => 55, 'oligarchy' => 65, 'theocracy' => 70, 'military_junta' => 50],
            ],
            'gov_invest_eff' => [
                'democracy' => ['default' => 1.0, 'heavy' => 0.95],
                'social_democracy' => ['default' => 1.05, 'heavy' => 0.9],
                'authoritarian' => ['default' => 1.0, 'heavy' => 1.08],
                'oligarchy' => ['default' => 0.95, 'heavy' => 1.1],
                'theocracy' => ['default' => 0.98, 'heavy' => 0.98],
                'military_junta' => ['default' => 0.95, 'heavy' => 1.12],
            ],
            'military' => [
                'base_recruit_rate' => 0.01,
                'readiness_gain_rate' => 1.5,
                'readiness_decay_rate' => 0.5,
                'equipment_factor' => 0.05,
                'tech_multiplier' => 0.02,
                'costs' => [
                    'c_manpower' => 0.02,
                    'c_equipment' => 0.04,
                    'c_readiness' => 0.05,
                    'status_mult' => [
                        'stationed' => 1.0,
                        'allied_base' => 1.2,
                        'deployed' => 1.5,
                        'combat' => 2.0,
                        'occupation' => 2.5,
                    ],
                ],
                'combat' => [
                    'mp_factor' => 1.0,
                    'eq_factor' => 1.3,
                    'readiness_factor' => 15.0,
                    'loyalty_factor' => 20.0,
                ],
                'loyalty' => ['lambda_loyalty' => 0.05],
            ],
            'prosperity' => [
                'alpha_p' => 0.8,
                'beta_p' => 0.8,
                'gamma_p' => 0.5,
                'delta_p' => 0.8,
                'epsilon_p' => 0.5,
                'c_baseline' => 1.0,
            ],
        ];
    }
}

if (!function_exists('apw_apply_gov_efficiency')) {
    function apw_apply_gov_efficiency(string $gov_type, string $bucket, float $value, array $cfg): float {
        $eff = $cfg['gov_invest_eff'][$gov_type][$bucket] ?? $cfg['gov_invest_eff'][$gov_type]['default'] ?? 1.0;
        return $value * $eff;
    }
}

if (!function_exists('apw_sigmoid')) {
    function apw_sigmoid(float $x): float {
        return 1.0 / (1.0 + exp(-$x));
    }
}

if (!function_exists('apw_process_turn')) {
    /**
     * Execute one deterministic turn.
     * @param array $state Current state (tiles, nations, turn, prices optional)
     * @param array $policies Map of nation_id => policy sliders (0–1 each)
     * @param array $config_override Override constants
     */
    function apw_process_turn(array $state, array $policies = [], array $config_override = []): array {
        $cfg = array_replace_recursive(apw_default_engine_config(), $config_override);
        $eps = $cfg['eps'];

        // Index nations by id for fast lookup.
        $nations = [];
        foreach ($state['nations'] ?? [] as $nation) {
            $nid = $nation['id'] ?? null;
            if ($nid === null) {
                continue;
            }
            $nations[$nid] = $nation;
        }

        // Precompute population per nation to allocate investments.
        $nation_pop = array_fill_keys(array_keys($nations), 0.0);
        foreach ($state['tiles'] ?? [] as $tile) {
            $nid = $tile['nation_id'] ?? null;
            if ($nid && isset($nation_pop[$nid])) {
                $nation_pop[$nid] += (float)($tile['population'] ?? 0.0);
            }
        }

        $updated_tiles = [];
        $avg_fsr_by_nation = array_fill_keys(array_keys($nations), 0.0);
        $tile_counts = array_fill_keys(array_keys($nations), 0);

        foreach ($state['tiles'] ?? [] as $tile) {
            $nid = $tile['nation_id'] ?? null;
            if (!$nid || !isset($nations[$nid])) {
                $updated_tiles[] = $tile;
                continue;
            }
            $nation = $nations[$nid];
            $gov_type = strtolower($nation['gov_type'] ?? 'democracy');
            $policy = $policies[$nid] ?? [];

            // Normalise policy vector if it exceeds 1.0 total.
            $policy_total = 0.0;
            foreach ($policy as $v) {
                $policy_total += max(0.0, (float)$v);
            }
            $scale = ($policy_total > 1.0) ? (1.0 / $policy_total) : 1.0;
            foreach ($policy as $k => $v) {
                $policy[$k] = max(0.0, (float)$v) * $scale;
            }

            $tile_pop = (float)($tile['population'] ?? 0.0);
            $nation_pop_total = max($eps, $nation_pop[$nid] ?? $eps);
            $budget = ($cfg['base_budget'] ?? 100.0) * (1.0 + $cfg['budget_lambda_industry'] * (float)($nation['avg_industry'] ?? 0.0));

            $corruption = (float)($nation['corruption_index'] ?? 0.0);
            $corruption_factor = max(0.0, 1.0 - $cfg['investment_loss_from_corruption'] * ($corruption / 100.0));

            $inv_raw = function (string $bucket) use ($budget, $policy, $tile_pop, $nation_pop_total): float {
                $p = $policy[$bucket] ?? 0.0;
                return $budget * $p * ($tile_pop / $nation_pop_total);
            };

            // Effective investments after government efficiency and corruption.
            $inv_agri = apw_apply_gov_efficiency($gov_type, 'default', $inv_raw('agriculture'), $cfg) * $corruption_factor;
            $inv_green = apw_apply_gov_efficiency($gov_type, 'default', $inv_raw('green_energy'), $cfg) * $corruption_factor;
            $inv_fossil = apw_apply_gov_efficiency($gov_type, 'default', $inv_raw('fossil_energy'), $cfg) * $corruption_factor;
            $inv_infra = apw_apply_gov_efficiency($gov_type, 'default', $inv_raw('infrastructure'), $cfg) * $corruption_factor;
            $inv_env = apw_apply_gov_efficiency($gov_type, 'default', $inv_raw('environmental_protection'), $cfg) * $corruption_factor;
            $inv_social = apw_apply_gov_efficiency($gov_type, 'default', $inv_raw('social_spending'), $cfg) * $corruption_factor;
            $inv_edu = apw_apply_gov_efficiency($gov_type, 'default', $inv_raw('education'), $cfg) * $corruption_factor;
            $inv_goods = apw_apply_gov_efficiency($gov_type, 'default', $inv_raw('industry_goods'), $cfg) * $corruption_factor;
            $inv_services = apw_apply_gov_efficiency($gov_type, 'default', $inv_raw('industry_services'), $cfg) * $corruption_factor;
            $inv_heavy = apw_apply_gov_efficiency($gov_type, 'heavy', $inv_raw('industry_heavy'), $cfg) * $corruption_factor;

            // Forestry/mining use industry buckets by default.
            $inv_forestry = $inv_goods;
            $inv_mining = $inv_heavy;

            // Extract nation techs.
            $tech_agr = (float)($nation['tech_agriculture'] ?? 0.0);
            $tech_energy = (float)($nation['tech_energy'] ?? 0.0);
            $tech_ind = (float)($nation['tech_industry'] ?? 0.0);
            $tech_infra = (float)($nation['tech_infrastructure'] ?? 0.0);
            $edu_index = (float)($nation['education_index'] ?? 0.0);

            $tconst = $cfg['tile_constants'];
            $biome = strtolower($tile['biome'] ?? $tile['terrain'] ?? 'plains');
            $biome_forest = $cfg['biome_factors'][$biome]['forest'] ?? 0.2;
            $biome_mineral = $cfg['biome_factors'][$biome]['mineral'] ?? 0.2;

            $tile_area = max($eps, (float)($tile['tile_area'] ?? 1.0));
            $env = (float)($tile['environment'] ?? 70.0);
            $poll = (float)($tile['pollution'] ?? 20.0);
            $food_prod = (float)($tile['food_production'] ?? 0.0);

            // 3.1 Food
            $food_need = $tconst['food']['a1'] * $tile_pop;
            $food_prod += $tconst['food']['alpha1'] * $inv_agri * (1.0 + $tconst['food']['eta1'] * $tech_agr) * ($env / 100.0);
            $food_prod -= $tconst['food']['alpha2'] * $poll;
            $food_prod = apw_clamp($food_prod, 0.0, 100.0);

            // 3.2 Energy
            $industry_total_prev = (float)($tile['industry_goods'] ?? 0.0)
                + (float)($tile['industry_services'] ?? 0.0)
                + (float)($tile['industry_heavy'] ?? 0.0);
            $energy_need = $tconst['energy']['b1'] * $tile_pop
                + $tconst['energy']['b2'] * $industry_total_prev
                + $tconst['energy']['b3'] * (float)($tile['infrastructure'] ?? 0.0);

            $renew_cap = (float)($tile['renewable_cap'] ?? 0.0);
            $renew_use = (float)($tile['renewable_use'] ?? 0.0);
            $renew_use = min(
                $renew_cap,
                $renew_use + $tconst['energy']['beta1'] * $inv_green * (1.0 + $tconst['energy']['eta2'] * $tech_energy)
            );

            $fossil_res = (float)($tile['fossil_reserves'] ?? $tile['nonrenew_reserves'] ?? 0.0);
            $fossil_extr = (float)($tile['fossil_extraction'] ?? $tile['nonrenew_use'] ?? 0.0);
            $fossil_demand = max(0.0, $energy_need - $renew_use);
            $fossil_extr = min(
                $fossil_res,
                $tconst['energy']['beta2'] * $inv_fossil * (1.0 + $tconst['energy']['eta3'] * $tech_energy)
                + $tconst['energy']['beta3'] * $fossil_demand
            );
            $fossil_res = max(0.0, $fossil_res - $fossil_extr);

            // 3.3 Resource extraction
            $forest_res = (float)($tile['forest_reserves'] ?? 0.0);
            $forest_extr = min($forest_res, $tconst['resources']['kappa1'] * $biome_forest * $inv_forestry * (1.0 + $tconst['resources']['eta_f'] * $tech_ind));
            $forest_res = max(0.0, $forest_res - $forest_extr);
            $env -= $tconst['resources']['lambda1'] * ($forest_extr / $tile_area);

            $mineral_res = (float)($tile['mineral_reserves'] ?? 0.0);
            $mining_extr = min($mineral_res, $tconst['resources']['kappa2'] * $biome_mineral * $inv_mining * (1.0 + $tconst['resources']['eta_m'] * $tech_ind));
            $mineral_res = max(0.0, $mineral_res - $mining_extr);
            $poll += $tconst['resources']['lambda2'] * $mining_extr;

            // 3.4 Industry updates
            $energy_sf = apw_clamp(
                apw_safe_div($renew_use + $fossil_extr, $energy_need + $eps, 0.0),
                0.0,
                1.0
            );
            $ind_goods = (float)($tile['industry_goods'] ?? 0.0);
            $ind_services = (float)($tile['industry_services'] ?? 0.0);
            $ind_heavy = (float)($tile['industry_heavy'] ?? 0.0);

            $ind_goods += $tconst['industry']['delta_g'] * $inv_goods * (1.0 + $tconst['industry']['eta_g'] * $tech_ind) * $energy_sf;
            $ind_goods -= $tconst['industry']['penalty_pollution_goods'] * $poll;

            $ind_services += $tconst['industry']['delta_s'] * $inv_services * (1.0 + $tconst['industry']['eta_s'] * $tech_ind) * $energy_sf * (1.0 + 0.01 * $edu_index);
            $ind_services -= $tconst['industry']['penalty_pollution_services'] * $poll;

            $ind_heavy += $tconst['industry']['delta_h'] * $inv_heavy * (1.0 + $tconst['industry']['eta_h'] * $tech_ind) * $energy_sf;
            $ind_heavy -= $tconst['industry']['penalty_pollution_heavy'] * $poll;

            $industry_total = max(0.0, $ind_goods + $ind_services + $ind_heavy);

            // 3.5 Environment & pollution
            $poll += $tconst['env']['phi1'] * $fossil_extr + $tconst['env']['phi2'] * $industry_total;
            $poll -= $tconst['env']['phi3'] * $inv_env * (1.0 + $tconst['env']['eta_env'] * $tech_energy);
            $poll = apw_clamp($poll, 0.0, 100.0);

            $env += $tconst['env']['psi1'] * $inv_env;
            $env -= $tconst['env']['psi2'] * $poll;
            $env = apw_clamp($env, 0.0, 100.0);

            // 3.6 Infrastructure
            $infra = (float)($tile['infrastructure'] ?? 0.0);
            $infra += $tconst['infra']['gamma1'] * $inv_infra * (1.0 + $tconst['infra']['eta4'] * $tech_infra);
            $infra -= $tconst['infra']['gamma2'] * $poll;
            $infra = apw_clamp($infra, 0.0, 100.0);

            // 3.7 Population
            $fsr = apw_clamp(apw_safe_div($food_prod, $food_need + $eps, 0.0), 0.0, $tconst['population']['fsr_cap']);
            $liv = $tconst['population']['w1'] * $fsr
                + $tconst['population']['w2'] * ($env / 100.0)
                + $tconst['population']['w3'] * (1.0 - $poll / 100.0)
                + $tconst['population']['w4'] * ($infra / 100.0);
            $tile_pop *= (1.0 + $tconst['population']['rho1'] * $liv - $tconst['population']['rho2'] * (1.0 - $liv));
            $tile_pop = max(0.0, $tile_pop);
            $density = apw_clamp(($tile_pop / $tile_area) * $cfg['k_dens'], 0.0, 100.0);

            $avg_fsr_by_nation[$nid] += $fsr;
            $tile_counts[$nid]++;

            $tile['population'] = $tile_pop;
            $tile['density'] = $density;
            $tile['food_production'] = $food_prod;
            $tile['food_need'] = $food_need;
            $tile['renewable_cap'] = $renew_cap;
            $tile['renewable_use'] = $renew_use;
            $tile['fossil_reserves'] = $fossil_res;
            $tile['fossil_extraction'] = $fossil_extr;
            $tile['energy_need'] = $energy_need;
            $tile['industry_goods'] = max(0.0, $ind_goods);
            $tile['industry_services'] = max(0.0, $ind_services);
            $tile['industry_heavy'] = max(0.0, $ind_heavy);
            $tile['environment'] = $env;
            $tile['pollution'] = $poll;
            $tile['infrastructure'] = $infra;
            $tile['forest_reserves'] = $forest_res;
            $tile['forest_extraction'] = $forest_extr;
            $tile['mineral_reserves'] = $mineral_res;
            $tile['mining_extraction'] = $mining_extr;
            $tile['biome'] = $biome;
            $updated_tiles[] = $tile;
        }

        // Aggregate per nation after tile updates.
        foreach ($nations as $nid => &$nation) {
            $nation_tiles = array_filter($updated_tiles, fn($t) => ($t['nation_id'] ?? null) === $nid);
            $tile_count = count($nation_tiles) ?: 1;
            $pop_total = 0.0;
            $food_prod_total = 0.0;
            $food_need_total = 0.0;
            $energy_prod_total = 0.0;
            $energy_need_total = 0.0;
            $goods_total = 0.0;
            $services_total = 0.0;
            $heavy_total = 0.0;
            $poll_sum = 0.0;
            $env_sum = 0.0;
            $infra_sum = 0.0;

            foreach ($nation_tiles as $t) {
                $pop_total += (float)$t['population'];
                $food_prod_total += (float)($t['food_production'] ?? 0.0);
                $food_need_total += (float)($t['food_need'] ?? 0.0);
                $energy_prod_total += (float)($t['renewable_use'] ?? 0.0) + (float)($t['fossil_extraction'] ?? 0.0);
                $energy_need_total += (float)($t['energy_need'] ?? 0.0);
                $goods_total += (float)($t['industry_goods'] ?? 0.0);
                $services_total += (float)($t['industry_services'] ?? 0.0);
                $heavy_total += (float)($t['industry_heavy'] ?? 0.0);
                $poll_sum += (float)($t['pollution'] ?? 0.0);
                $env_sum += (float)($t['environment'] ?? 0.0);
                $infra_sum += (float)($t['infrastructure'] ?? 0.0);
            }

            $nation['population_prev'] = $nation['population_total'] ?? $pop_total;
            $nation['population_total'] = $pop_total;
            $nation['food_production_total'] = $food_prod_total;
            $nation['food_need_total'] = $food_need_total;
            $nation['energy_production_total'] = $energy_prod_total;
            $nation['energy_need_total'] = $energy_need_total;
            $nation['goods_output'] = $goods_total;
            $nation['services_output'] = $services_total;
            $nation['heavy_output'] = $heavy_total;
            $nation['avg_pollution'] = $poll_sum / $tile_count;
            $nation['avg_environment'] = $env_sum / $tile_count;
            $nation['avg_infrastructure'] = $infra_sum / $tile_count;
            $nation['avg_food_security'] = apw_safe_div($avg_fsr_by_nation[$nid], max(1, $tile_counts[$nid]), 0.0);
            $nation['avg_energy_security'] = apw_safe_div($energy_prod_total, $energy_need_total + $eps, 0.0);
        }
        unset($nation);

        // Military updates before indices adjust econ balance.
        foreach ($nations as $nid => &$nation) {
            $policy = $policies[$nid] ?? [];
            $mil_cfg = $cfg['military'];
            $pop_loyalty = apw_clamp((float)($nation['population_loyalty'] ?? 0.5), 0.0, 1.0);
            $corruption = (float)($nation['corruption_index'] ?? 0.0);
            $loyalty_programs = max(0.0, min(1.0, (float)($policy['loyalty_programs'] ?? 0.0)));
            $pop_loyalty += $mil_cfg['loyalty']['lambda_loyalty'] * ($loyalty_programs - $corruption / 100.0);
            $pop_loyalty = apw_clamp($pop_loyalty, 0.0, 1.0);
            $nation['population_loyalty'] = $pop_loyalty;

            $recruits_available = $mil_cfg['base_recruit_rate'] * ($nation['population_total'] ?? 0.0) * $pop_loyalty;
            $recruitment_policy = max(0.0, min(1.0, (float)($policy['recruitment_policy'] ?? 0.0)));
            $recruits_intake = $recruits_available * $recruitment_policy;
            $ground_frac = max(0.0, min(1.0, (float)($policy['military_ground'] ?? 0.0)));
            $air_frac = max(0.0, min(1.0, (float)($policy['military_air'] ?? 0.0)));
            $naval_frac = max(0.0, min(1.0, (float)($policy['military_naval'] ?? 0.0)));
            $frac_total = max($eps, $ground_frac + $air_frac + $naval_frac);
            $ground_frac /= $frac_total;
            $air_frac /= $frac_total;
            $naval_frac /= $frac_total;

            $ground_forces = $nation['ground_forces'] ?? ['manpower' => 0.0, 'equipment_score' => 0.0, 'readiness' => 0.0, 'status' => 'stationed'];
            $air_forces = $nation['air_forces'] ?? ['manpower' => 0.0, 'equipment_score' => 0.0, 'readiness' => 0.0, 'status' => 'stationed'];
            $naval_forces = $nation['naval_forces'] ?? ['manpower' => 0.0, 'equipment_score' => 0.0, 'readiness' => 0.0, 'status' => 'stationed'];

            $ground_forces['manpower'] += $recruits_intake * $ground_frac;
            $air_forces['manpower'] += $recruits_intake * $air_frac;
            $naval_forces['manpower'] += $recruits_intake * $naval_frac;

            $readiness_spend = max(0.0, min(1.0, (float)($policy['readiness_spending'] ?? 0.0)));
            $apply_readiness = function (array $forces, float $tech): array use ($mil_cfg, $readiness_spend) {
                $forces['readiness'] += $mil_cfg['readiness_gain_rate'] * $readiness_spend * (1.0 + $tech / 100.0);
                $forces['readiness'] -= $mil_cfg['readiness_decay_rate'];
                $forces['readiness'] = apw_clamp($forces['readiness'], 0.0, 100.0);
                return $forces;
            };
            $ground_forces = $apply_readiness($ground_forces, (float)($nation['tech_military_ground'] ?? 0.0));
            $air_forces = $apply_readiness($air_forces, (float)($nation['tech_military_air'] ?? 0.0));
            $naval_forces = $apply_readiness($naval_forces, (float)($nation['tech_military_naval'] ?? 0.0));

            $equip_gain = function (float $heavy_output, float $frac) use ($mil_cfg): float {
                return $mil_cfg['equipment_factor'] * $heavy_output * $frac;
            };
            $ground_forces['equipment_score'] += $equip_gain($nation['heavy_output'] ?? 0.0, $ground_frac);
            $air_forces['equipment_score'] += $equip_gain($nation['heavy_output'] ?? 0.0, $air_frac);
            $naval_forces['equipment_score'] += $equip_gain($nation['heavy_output'] ?? 0.0, $naval_frac);

            $tech_mult = $mil_cfg['tech_multiplier'];
            $ground_forces['equipment_score'] *= (1.0 + $tech_mult * ($nation['tech_military_ground'] ?? 0.0));
            $air_forces['equipment_score'] *= (1.0 + $tech_mult * ($nation['tech_military_air'] ?? 0.0));
            $naval_forces['equipment_score'] *= (1.0 + $tech_mult * ($nation['tech_military_naval'] ?? 0.0));

            $cost_calc = function (array $forces) use ($mil_cfg): float {
                $base = $forces['manpower'] * $mil_cfg['costs']['c_manpower']
                    + $forces['equipment_score'] * $mil_cfg['costs']['c_equipment']
                    + $forces['readiness'] * $mil_cfg['costs']['c_readiness'];
                $status = $forces['status'] ?? 'stationed';
                $mult = $mil_cfg['costs']['status_mult'][$status] ?? 1.0;
                return $base * $mult;
            };
            $mil_cost = $cost_calc($ground_forces) + $cost_calc($air_forces) + $cost_calc($naval_forces);
            $nation['military_cost'] = $mil_cost;
            $nation['ground_forces'] = $ground_forces;
            $nation['air_forces'] = $air_forces;
            $nation['naval_forces'] = $naval_forces;
        }
        unset($nation);

        // National indices and economic balance.
        foreach ($nations as $nid => &$nation) {
            $policy = $policies[$nid] ?? [];
            $econ_cfg = $cfg['econ_balance'];
            $food_surplus = ($nation['food_production_total'] ?? 0.0) - ($nation['food_need_total'] ?? 0.0);
            $energy_surplus = ($nation['energy_production_total'] ?? 0.0) - ($nation['energy_need_total'] ?? 0.0);
            $nation['food_surplus'] = $food_surplus;
            $nation['energy_surplus'] = $energy_surplus;

            $econ_bal = $econ_cfg['chi1'] * $food_surplus
                + $econ_cfg['chi2'] * $energy_surplus
                + $econ_cfg['chi3'] * (($nation['goods_output'] ?? 0.0) + ($nation['services_output'] ?? 0.0) + ($nation['heavy_output'] ?? 0.0))
                - $econ_cfg['chi4'] * ($nation['avg_pollution'] ?? 0.0);
            $econ_bal -= ($nation['military_cost'] ?? 0.0);
            $nation['econ_balance'] = $econ_bal;

            // Standard of living
            $pros_cfg = $cfg['prosperity'];
            $pros_per_cap = $pros_cfg['alpha_p'] * apw_safe_div((float)$nation['goods_output'] + (float)$nation['services_output'], ($nation['population_total'] ?? 0.0) + $eps, 0.0)
                + $pros_cfg['beta_p'] * ($nation['avg_food_security'] ?? 0.0)
                + $pros_cfg['gamma_p'] * ($nation['avg_energy_security'] ?? 0.0)
                + $pros_cfg['delta_p'] * apw_safe_div((float)($nation['tech_medicine'] ?? 0.0), $cfg['tech_max'], 0.0)
                + $pros_cfg['epsilon_p'] * apw_safe_div(($nation['avg_infrastructure'] ?? 0.0), 100.0, 0.0);
            $nation['standard_of_living'] = 100.0 * apw_sigmoid($pros_per_cap - $pros_cfg['c_baseline']);

            // Education
            $edu_cfg = $cfg['education'];
            $pop_prev = max($eps, (float)($nation['population_prev'] ?? $nation['population_total'] ?? 0.0));
            $pop_growth_strain = max(0.0, (($nation['population_total'] ?? 0.0) - $pop_prev) / $pop_prev - $edu_cfg['pop_strain_threshold']);
            $edu = (float)($nation['education_index'] ?? 0.0);
            $edu += $edu_cfg['theta1'] * ($policy['education'] ?? 0.0) * (1.0 + $edu_cfg['sigma1'] * ($policy['social_spending'] ?? 0.0));
            $edu -= $edu_cfg['theta2'] * $pop_growth_strain;
            $nation['education_index'] = apw_clamp($edu, 0.0, 100.0);

            // Tech growth
            $tg = $cfg['tech_growth'];
            $tech_fields = [
                'tech_agriculture' => 'agriculture',
                'tech_industry' => 'industry',
                'tech_energy' => 'energy',
                'tech_infrastructure' => 'infrastructure',
                'tech_medicine' => 'medicine',
                'tech_military_ground' => 'military_ground',
                'tech_military_air' => 'military_air',
                'tech_military_naval' => 'military_naval',
            ];
            foreach ($tech_fields as $field => $key) {
                $relevant_policy = match ($key) {
                    'agriculture' => ($policy['agriculture'] ?? 0.0),
                    'industry' => ($policy['industry_goods'] ?? 0.0) + ($policy['industry_services'] ?? 0.0) + ($policy['industry_heavy'] ?? 0.0),
                    'energy' => ($policy['green_energy'] ?? 0.0) + 0.5 * ($policy['fossil_energy'] ?? 0.0),
                    'infrastructure' => ($policy['infrastructure'] ?? 0.0),
                    'medicine' => ($policy['social_spending'] ?? 0.0),
                    'military_ground' => ($policy['military_ground'] ?? 0.0),
                    'military_air' => ($policy['military_air'] ?? 0.0),
                    'military_naval' => ($policy['military_naval'] ?? 0.0),
                    default => 0.0,
                };
                $tech_val = (float)($nation[$field] ?? 0.0);
                $tech_val += ($tg['tau'][$key] ?? 0.05) * $relevant_policy * ($tg['base_research'] + $tg['sigma2'] * ($nation['education_index'] ?? 0.0) / 100.0);
                $nation[$field] = apw_clamp($tech_val, 0.0, $cfg['tech_max']);
            }

            // Media
            $media_cfg = $cfg['media'];
            $media = (float)($nation['media_freedom'] ?? 50.0);
            $media = (1.0 - $media_cfg['omega']) * $media + $media_cfg['omega'] * ($policy['media_support'] ?? 0.0) * 100.0;
            $media_cap = $media_cfg['caps'][strtolower($nation['gov_type'] ?? 'democracy')] ?? 100.0;
            $media = min($media, $media_cap);
            $nation['media_freedom'] = apw_clamp($media, 0.0, 100.0);

            // Corruption
            $corr_cfg = $cfg['corruption'];
            $inequality = 1.0 - ($nation['standard_of_living'] ?? 0.0) / 100.0 * (($policy['social_spending'] ?? 0.0) + ($policy['environmental_protection'] ?? 0.0)) / 2.0;
            $corr = (float)($nation['corruption_index'] ?? 0.0);
            $corr += $corr_cfg['c1'] * $inequality;
            $corr -= $corr_cfg['c2'] * ($policy['anti_corruption'] ?? 0.0);
            $corr -= $corr_cfg['c3'] * ($nation['education_index'] ?? 0.0) / 100.0;
            $corr -= $corr_cfg['c4'] * ($nation['media_freedom'] ?? 0.0) / 100.0;
            $corr += $corr_cfg['bias'][strtolower($nation['gov_type'] ?? 'democracy')] ?? 0.0;
            $nation['corruption_index'] = apw_clamp($corr, 0.0, 100.0);

            // Happiness
            $hap_cfg = $cfg['happiness'];
            $happ_base = 100.0 * (
                $hap_cfg['h1'] * ($nation['avg_food_security'] ?? 0.0)
                + $hap_cfg['h2'] * (($nation['avg_environment'] ?? 0.0) / 100.0)
                + $hap_cfg['h3'] * (1.0 - ($nation['avg_pollution'] ?? 0.0) / 100.0)
                + $hap_cfg['h4'] * (($nation['avg_infrastructure'] ?? 0.0) / 100.0)
                + $hap_cfg['h5'] * ($policy['social_spending'] ?? 0.0)
            );
            // Government-style adjustments
            $gov_type = strtolower($nation['gov_type'] ?? 'democracy');
            if ($gov_type === 'authoritarian' && ($nation['standard_of_living'] ?? 0.0) > 60 && ($nation['media_freedom'] ?? 0.0) < 40) {
                $happ_base += 3.0;
            }
            if ($gov_type === 'authoritarian' && ($nation['standard_of_living'] ?? 0.0) < 40 && ($nation['media_freedom'] ?? 0.0) < 40) {
                $happ_base -= 8.0;
            }
            if ($gov_type === 'military_junta' && ($nation['avg_pollution'] ?? 0.0) > 60) {
                $happ_base -= 5.0;
            }
            $happ_base -= 0.1 * ($nation['corruption_index'] ?? 0.0);
            $nation['happiness'] = apw_clamp($happ_base, 0.0, 100.0);

            // Combat power snapshot
            $combat_cfg = $cfg['military']['combat'];
            $calc_power = function (array $forces, float $tech, float $loyalty) use ($combat_cfg): float {
                $power = ($forces['manpower'] ?? 0.0) * $combat_cfg['mp_factor']
                    + ($forces['equipment_score'] ?? 0.0) * $combat_cfg['eq_factor']
                    + ($forces['readiness'] ?? 0.0) / 100.0 * $combat_cfg['readiness_factor']
                    + $loyalty * $combat_cfg['loyalty_factor'];
                return $power * (1.0 + $tech / 10.0);
            };
            $nation['combat_power'] = [
                'ground' => $calc_power($nation['ground_forces'], (float)($nation['tech_military_ground'] ?? 0.0), $nation['population_loyalty']),
                'air' => $calc_power($nation['air_forces'], (float)($nation['tech_military_air'] ?? 0.0), $nation['population_loyalty']),
                'naval' => $calc_power($nation['naval_forces'], (float)($nation['tech_military_naval'] ?? 0.0), $nation['population_loyalty']),
            ];
        }
        unset($nation);

        // Assemble new state
        $next_turn = ((int)($state['turn'] ?? 0)) + 1;
        return [
            'turn' => $next_turn,
            'tiles' => $updated_tiles,
            'nations' => array_values($nations),
            'summary' => [
                'updated_at' => date('c'),
                'note' => 'Processed turn with deterministic APW engine.',
            ],
        ];
    }
}
