<?php
// gameapp/map_generator.php
// Endpoint to preview procedural maps before committing to map_turns.
require __DIR__ . '/auth.php';
require_once __DIR__ . '/world_map.php';

header('Content-Type: application/json');

$user = require_login();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'POST required']);
    exit;
}

$allowed_map_types       = ['standard', 'continents', 'pangea', 'archipelago'];
$allowed_map_sizes       = ['small', 'medium', 'large', 'huge'];
$allowed_resolutions     = ['starter', 'standard', 'advanced', 'district', 'enterprise'];
$allowed_presets         = ['earth_like', 'fragmented_world', 'ring_world'];
$allowed_land_sea_styles = ['balanced', 'land_heavy', 'sea_heavy', 'polar_caps'];

$resolution_map = ['starter', 'standard', 'advanced', 'district', 'enterprise'];

// Enforce account-based resolution caps server-side.
$pdo = get_pdo();
$stmt = $pdo->prepare(
    'SELECT at.name AS account_name
     FROM users u
     LEFT JOIN account_types at ON u.account_type_id = at.id
     WHERE u.id = ?'
);
$stmt->execute([$user['id']]);
$acct = $stmt->fetch();
$acct_name = strtolower($acct['account_name'] ?? '');
$resolution_cap = 'standard';
if (str_contains($acct_name, 'district') || str_contains($acct_name, 'enterprise')) {
    $resolution_cap = 'enterprise';
} elseif (str_contains($acct_name, 'advanced') || str_contains($acct_name, 'pro')) {
    $resolution_cap = 'advanced';
}
$cap_index = max(0, array_search($resolution_cap, $resolution_map, true));

$map_type_raw      = strtolower(trim($_POST['map_type'] ?? 'standard'));
$map_type          = in_array($map_type_raw, $allowed_map_types, true) ? $map_type_raw : 'standard';
$map_size_raw      = strtolower(trim($_POST['map_size'] ?? 'medium'));
$map_size          = in_array($map_size_raw, $allowed_map_sizes, true) ? $map_size_raw : 'medium';
$preset_map_raw    = strtolower(trim($_POST['preset_map'] ?? ''));
$preset_map        = in_array($preset_map_raw, $allowed_presets, true) ? $preset_map_raw : '';
$land_sea_style    = strtolower(trim($_POST['land_sea_pattern'] ?? 'balanced'));
$land_sea_pattern  = in_array($land_sea_style, $allowed_land_sea_styles, true) ? $land_sea_style : 'balanced';

$land_ratio       = isset($_POST['land_sea_ratio']) ? max(0.1, min(0.9, (float)$_POST['land_sea_ratio'])) : 0.5;
$lacunarity       = isset($_POST['lacunarity']) ? max(0.6, min(3.0, (float)$_POST['lacunarity'])) : 1.5;
$inequality_raw   = isset($_POST['inequality_slider']) ? (float)$_POST['inequality_slider'] : 0.5;
$inequality       = ($inequality_raw < 0.34) ? 'low' : (($inequality_raw > 0.67) ? 'high' : 'medium');
$resolution_raw   = trim((string)($_POST['resolution_level'] ?? 'standard'));
if (ctype_digit($resolution_raw)) {
    $resolution_idx = max(0, min(count($resolution_map) - 1, ((int)$resolution_raw) - 1));
    $resolution_raw = $resolution_map[$resolution_idx];
}
$resolution_level = in_array(strtolower($resolution_raw), $allowed_resolutions, true)
    ? strtolower($resolution_raw)
    : 'standard';
$resolution_level_indexed = max(0, array_search($resolution_level, $resolution_map, true));
if ($resolution_level_indexed > $cap_index) {
    $resolution_level = $resolution_map[$cap_index];
}
$num_nations      = isset($_POST['num_nations']) ? (int)$_POST['num_nations'] : 4;
$num_nations      = max(2, min(18, $num_nations));
$seed_raw         = trim((string)($_POST['seed'] ?? ''));
$seed             = $seed_raw !== '' ? substr($seed_raw, 0, 64) : ($user['id'] . '-' . time());

$opts = [
    'seed' => $seed,
    'game_id' => $_POST['game_id'] ?? null,
    'map_type' => $map_type,
    'map_size' => $map_size,
    'inequality_level' => $inequality,
    'land_sea_pattern' => $land_sea_pattern,
    'preset_map' => $preset_map,
    'num_nations' => $num_nations,
    'land_sea_ratio' => $land_ratio,
    'lacunarity' => $lacunarity,
    'resolution_level' => $resolution_level,
];

$map = apw_generate_world_map($opts);
$html = apw_render_map_html($map);

$response = [
    'html' => $html,
    'summary' => sprintf(
        'Seed %s • %dx%d tiles • %s map, %s size, %d nations • Land %.0f%%, lacunarity %.2f, inequality %s',
        $map['seed'],
        $map['width'],
        $map['height'],
        ucfirst($map['map_type']),
        ucfirst($map['map_size']),
        (int)$opts['num_nations'],
        $land_ratio * 100,
        $lacunarity,
        ucfirst($inequality)
    ),
    'seed' => $map['seed'],
    'meta' => [
        'land_sea_ratio' => $land_ratio,
        'lacunarity' => $lacunarity,
        'inequality_level' => $inequality,
        'resolution_level' => $resolution_level,
    ],
];

echo json_encode($response);
exit;
