<?php
// gameapp/map_generation.php
// Standalone procedural map generator for teacher-driven drafts.

if (!function_exists('apw_map_dimensions')) {
    /**
     * Convert human-readable size into grid dimensions.
     */
    function apw_map_dimensions(string $size, string $resolution_level = 'standard'): array {
        $size = strtolower($size);
        $resolution_level = strtolower($resolution_level);

        // Resolution level acts as a cap on map density.
        $resolution_caps = [
            'starter'    => 0.7,
            'standard'   => 1.0,
            'advanced'   => 1.2,
            'district'   => 1.4,
            'enterprise' => 1.7,
        ];
        $cap = $resolution_caps[$resolution_level] ?? 1.0;

        switch ($size) {
            case 'small':
                return [max(12, (int)round(28 * $cap)), max(8, (int)round(18 * $cap))];
            case 'large':
                return [max(16, (int)round(56 * $cap)), max(12, (int)round(36 * $cap))];
            case 'huge':
                return [max(24, (int)round(72 * $cap)), max(16, (int)round(46 * $cap))];
            case 'medium':
            default:
                return [max(14, (int)round(40 * $cap)), max(10, (int)round(24 * $cap))];
        }
    }
}

if (!function_exists('apw_hash_noise')) {
    /**
     * Deterministic pseudo-random value in [0,1) derived from a seed and coordinates.
     */
    function apw_hash_noise(string $seed, int $x, int $y, int $scale = 1): float {
        $sx = intdiv($x, max(1, $scale));
        $sy = intdiv($y, max(1, $scale));
        $hash = crc32($seed . '|' . $sx . '|' . $sy . '|' . $scale);
        return ($hash & 0xffff) / 0xffff;
    }
}

if (!function_exists('apw_blended_height')) {
    /**
     * Layer a few noise octaves to get smooth-ish elevation with adjustable lacunarity.
     */
    function apw_blended_height(string $seed, int $x, int $y, float $lacunarity = 1.5): float {
        $lacunarity = max(0.6, min(3.0, $lacunarity));
        $n1 = apw_hash_noise($seed . 'a', (int)($x * $lacunarity), (int)($y * $lacunarity), 3);
        $n2 = apw_hash_noise($seed . 'b', (int)($x * $lacunarity * 0.6), (int)($y * $lacunarity * 0.6), 7);
        $n3 = apw_hash_noise($seed . 'c', (int)($x * $lacunarity * 0.3), (int)($y * $lacunarity * 0.3), 13);
        return ($n1 * 0.55) + ($n2 * 0.3) + ($n3 * 0.15);
    }
}

if (!function_exists('apw_generate_world_map')) {
    /**
     * Build a deterministic map using game options. Output includes tiles, nations, and cities.
     */
    function apw_generate_world_map(array $opts): array {
        $seed         = $opts['seed'] ?? ($opts['game_id'] ?? 'seed');
        $map_type     = strtolower($opts['map_type'] ?? 'standard');
        $map_size     = strtolower($opts['map_size'] ?? 'medium');
        $inequality   = strtolower($opts['inequality_level'] ?? 'medium');
        $land_bias    = isset($opts['land_sea_ratio']) ? (float)$opts['land_sea_ratio'] : 0.5; // 0 water heavy, 1 land heavy
        $lacunarity   = isset($opts['lacunarity']) ? (float)$opts['lacunarity'] : 1.5;
        $resolution   = strtolower($opts['resolution_level'] ?? 'standard');
        $preset_map   = strtolower($opts['preset_map'] ?? '');
        $num_nations  = max(1, (int)($opts['num_nations'] ?? 4));
        $land_pattern = strtolower($opts['land_sea_pattern'] ?? 'balanced');

        [$width, $height] = apw_map_dimensions($map_size, $resolution);
        $tiles = [];

        // Base thresholds
        $water_bias = 1.0 - max(0.05, min(0.95, $land_bias));
        if ($map_type === 'archipelago') {
            $water_bias += 0.08;
        } elseif ($map_type === 'pangea') {
            $water_bias -= 0.08;
        }
        if ($land_pattern === 'land_heavy') {
            $water_bias -= 0.08;
        } elseif ($land_pattern === 'sea_heavy') {
            $water_bias += 0.08;
        }
        if ($preset_map === 'earth_like') {
            $water_bias = 0.52;
        } elseif ($preset_map === 'fragmented_world') {
            $water_bias = min(0.72, $water_bias + 0.05);
        }

        $resource_bias = 0.5;
        if ($inequality === 'low') {
            $resource_bias = 0.45;
        } elseif ($inequality === 'high') {
            $resource_bias = 0.6;
        }

        $height_map = [];
        for ($y = 0; $y < $height; $y++) {
            $row = [];
            for ($x = 0; $x < $width; $x++) {
                $height_val = apw_blended_height($seed, $x, $y, $lacunarity);

                // Radial falloff keeps edges watery for nicer coastlines.
                $dx = ($x / max(1, $width - 1)) - 0.5;
                $dy = ($y / max(1, $height - 1)) - 0.5;
                $dist = sqrt($dx * $dx + $dy * $dy);
                $height_val -= $dist * 0.18;

                if ($land_pattern === 'polar_caps') {
                    $height_val -= max(0, (abs($dy) - 0.12) * 0.35);
                }

                if ($map_type === 'continents') {
                    $height_val += 0.06 * cos($x / max(1, $width - 1) * M_PI * 2);
                } elseif ($map_type === 'pangea') {
                    $height_val += 0.08;
                }

                if ($map_type === 'archipelago') {
                    $height_val -= 0.04 * sin($y / max(1, $height - 1) * M_PI * 2);
                }

                if ($preset_map === 'ring_world') {
                    $ring_dist = abs($dist - 0.25);
                    $height_val -= $ring_dist * 0.45;
                }

                $moisture = apw_hash_noise($seed . 'm', $x, $y, 5);
                $resources = apw_hash_noise($seed . 'r', $x, $y, 4);
                $height_map[$y][$x] = $height_val;

                $row[] = [
                    'x' => $x,
                    'y' => $y,
                    'terrain' => 'water',
                    'height' => round($height_val, 3),
                    'moisture' => round($moisture, 3),
                    'resource_score' => round(($resources * 0.5) + ($resource_bias * 0.5), 3),
                    'nation_id' => null,
                    'population' => 0,
                    'city' => null,
                    'border' => false,
                ];
            }
            $tiles[] = $row;
        }

        // Terrain classification
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                $height_val = $height_map[$y][$x];
                $is_land = $height_val > $water_bias;
                $moisture = $tiles[$y][$x]['moisture'];

                $terrain = 'water';
                if ($is_land) {
                    if ($height_val > $water_bias + 0.28) {
                        $terrain = 'mountain';
                    } elseif ($height_val > $water_bias + 0.16) {
                        $terrain = 'hill';
                    } else {
                        if ($moisture < 0.2) {
                            $terrain = 'desert';
                        } elseif ($moisture < 0.55) {
                            $terrain = 'plains';
                        } elseif ($moisture < 0.78) {
                            $terrain = 'forest';
                        } else {
                            $terrain = 'wetland';
                        }
                    }
                }
                $tiles[$y][$x]['terrain'] = $terrain;
            }
        }

        // Coast detection for water tiles bordering land.
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                if ($tiles[$y][$x]['terrain'] === 'water') {
                    $neighbors = [[$x - 1, $y], [$x + 1, $y], [$x, $y - 1], [$x, $y + 1]];
                    foreach ($neighbors as [$nx, $ny]) {
                        if ($nx >= 0 && $nx < $width && $ny >= 0 && $ny < $height) {
                            if ($tiles[$ny][$nx]['terrain'] !== 'water') {
                                $tiles[$y][$x]['terrain'] = 'coast';
                                break;
                            }
                        }
                    }
                }
            }
        }

        // Assign nations via seeded Voronoi cells.
        $nation_seeds = [];
        $attempts = 0;
        while (count($nation_seeds) < $num_nations && $attempts < $width * $height) {
            $rx = random_int(0, $width - 1);
            $ry = random_int(0, $height - 1);
            if (in_array($tiles[$ry][$rx]['terrain'], ['water', 'coast'], true)) {
                $attempts++;
                continue;
            }
            $nation_seeds[] = [$rx, $ry];
            $attempts++;
        }

        $nation_palette = ['#c0392b', '#16a085', '#8e44ad', '#2980b9', '#d35400', '#27ae60', '#7f8c8d', '#f39c12'];
        $nations = [];
        foreach ($nation_seeds as $idx => $seed_xy) {
            $nations[] = [
                'id' => $idx + 1,
                'name' => 'Nation ' . ($idx + 1),
                'color' => $nation_palette[$idx % count($nation_palette)],
                'seed' => $seed_xy,
                'wealth_bias' => max(0.4, min(1.6, 1 + ($resource_bias - 0.5) * (0.8 + $idx * 0.05))),
            ];
        }

        // Territory assignment
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                if (in_array($tiles[$y][$x]['terrain'], ['water', 'coast'], true)) {
                    continue;
                }
                $closest = null;
                $closest_d = PHP_FLOAT_MAX;
                foreach ($nations as $nation) {
                    [$nx, $ny] = $nation['seed'];
                    $d = abs($nx - $x) + abs($ny - $y);
                    if ($d < $closest_d) {
                        $closest = $nation['id'];
                        $closest_d = $d;
                    }
                }
                $tiles[$y][$x]['nation_id'] = $closest;
            }
        }

        // Border detection
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                $tile = &$tiles[$y][$x];
                if ($tile['nation_id'] === null) {
                    continue;
                }
                $neighbors = [[$x - 1, $y], [$x + 1, $y], [$x, $y - 1], [$x, $y + 1]];
                foreach ($neighbors as [$nx, $ny]) {
                    if ($nx >= 0 && $nx < $width && $ny >= 0 && $ny < $height) {
                        $other = $tiles[$ny][$nx];
                        if ($other['nation_id'] !== $tile['nation_id'] && $other['nation_id'] !== null) {
                            $tile['border'] = true;
                            break;
                        }
                    }
                }
            }
        }

        // Population distribution and cities
        $city_names = [];
        for ($y = 0; $y < $height; $y++) {
            for ($x = 0; $x < $width; $x++) {
                $tile = &$tiles[$y][$x];
                if ($tile['nation_id'] === null) {
                    continue;
                }
                $terrain_factor = match ($tile['terrain']) {
                    'plains' => 1.0,
                    'forest' => 0.9,
                    'wetland' => 0.8,
                    'hill' => 0.6,
                    'mountain' => 0.2,
                    'desert' => 0.3,
                    default => 0.1,
                };
                $inequality_spread = match ($inequality) {
                    'low' => 0.1,
                    'high' => 0.6,
                    default => 0.35,
                };
                $wealth_roll = apw_hash_noise($seed . 'w', $x, $y, 3);
                $population = max(0, ($tile['resource_score'] * 0.6 + $terrain_factor * 0.4))
                    * (1 + ($wealth_roll - 0.5) * $inequality_spread);
                $tile['population'] = round($population, 3);
            }
        }

        // Identify top population centers per nation
        $cities = [];
        foreach ($nations as $nation) {
            $nation_tiles = [];
            for ($y = 0; $y < $height; $y++) {
                for ($x = 0; $x < $width; $x++) {
                    if ($tiles[$y][$x]['nation_id'] === $nation['id']) {
                        $nation_tiles[] = $tiles[$y][$x];
                    }
                }
            }
            usort($nation_tiles, fn($a, $b) => $b['population'] <=> $a['population']);
            $city_count = max(1, min(3, (int)ceil(count($nation_tiles) / 150)));
            for ($i = 0; $i < $city_count && isset($nation_tiles[$i]); $i++) {
                $c_tile = $nation_tiles[$i];
                $city_name = 'City ' . $nation['id'] . '-' . ($i + 1);
                $cities[] = [
                    'name' => $city_name,
                    'nation_id' => $nation['id'],
                    'x' => $c_tile['x'],
                    'y' => $c_tile['y'],
                    'population' => $c_tile['population'],
                ];
                $tiles[$c_tile['y']][$c_tile['x']]['city'] = $city_name;
            }
        }

        return [
            'width'  => $width,
            'height' => $height,
            'seed'   => $seed,
            'map_type' => $map_type,
            'map_size' => $map_size,
            'resolution_level' => $resolution,
            'inequality_level' => $inequality,
            'land_sea_ratio' => $land_bias,
            'lacunarity' => $lacunarity,
            'tiles'  => $tiles,
            'nations' => $nations,
            'cities' => $cities,
        ];
    }
}
?>
