<?php
// gameapp/game.php
require __DIR__ . '/auth.php'; // for get_pdo()
$pdo = get_pdo();

function apw_safe_get(array $arr, string $key, $default = null) {
    return $arr[$key] ?? $default;
}

function apw_find_nation_state(array $game_state, int $nation_id): ?array {
    foreach ($game_state['nations'] ?? [] as $nation) {
        if ((int)($nation['id'] ?? 0) === $nation_id) {
            return $nation;
        }
    }
    return null;
}

// --- Find student by code ----------------------------------------------------
$code = $_GET['s'] ?? '';
$code = trim($code);

if ($code === '') {
    http_response_code(400);
    echo 'Missing student code.';
    exit;
}

$stmt = $pdo->prepare(
    'SELECT
         s.id              AS student_id,
         s.label           AS student_label,
         s.student_code,
         s.game_id         AS student_game_id,
         u.email           AS teacher_email,
         g.id              AS game_id,
         g.name            AS game_name,
         g.map_type,
         g.map_size,
         g.inequality_level,
         g.land_sea_pattern,
         g.preset_map,
         g.game_state,
         g.allow_student_team_edits,
         g.created_at      AS game_created_at,
         g.current_turn,
         g.turn_mode,
         g.next_turn_at,
         n.id              AS nation_id,
         n.name            AS nation_name,
         n.primary_color,
         n.secondary_color,
         n.stripe_style,
         n.stripe_width_fraction,
         ps.name           AS political_name,
         es.name           AS economic_name,
         mf.name           AS media_name
     FROM students s
     LEFT JOIN users u            ON u.id = s.teacher_id
     LEFT JOIN games g            ON g.id = s.game_id
     LEFT JOIN nation_members nm  ON nm.student_id = s.id
     LEFT JOIN nations n          ON n.id = nm.nation_id
     LEFT JOIN political_systems ps ON ps.id = n.political_system_id
     LEFT JOIN economic_systems  es ON es.id = n.economic_system_id
     LEFT JOIN media_freedom     mf ON mf.id = n.media_freedom_id
     WHERE s.student_code = ?
     LIMIT 1'
);
$stmt->execute([$code]);
$data = $stmt->fetch();

if (!$data) {
    http_response_code(404);
    echo 'Student not found. Check your link or ask your teacher for a new code.';
    exit;
}

$student_label  = $data['student_label'] ?? 'Student';
$teacher_email  = $data['teacher_email'] ?? 'your teacher';
$game_id        = (int)($data['game_id'] ?? 0);
$game_name      = $data['game_name'] ?? null;
$nation_id      = (int)($data['nation_id'] ?? 0);
$game_state_raw = $data['game_state'] ?? '';
$game_state     = [];
if ($game_state_raw) {
    $decoded = json_decode($game_state_raw, true);
    if (is_array($decoded)) {
        $game_state = $decoded;
    }
}
$nation_state = $nation_id ? apw_find_nation_state($game_state, $nation_id) : null;

if ($game_id === 0 || !$game_name) {
    ?>
    <!doctype html>
    <html>
    <head>
        <meta charset="utf-8">
        <title>A Possible World – Game</title>
        <link rel="stylesheet" href="style.css">
    </head>
    <body class="app-body">
    <div class="app-shell">
        <header class="app-header">
            <div>
                <div class="app-title">A Possible World</div>
                <div class="app-subtitle">Hi, <?php echo htmlspecialchars($student_label); ?>!</div>
            </div>
        </header>
        <main class="app-main">
            <div class="card">
                <h2>Not assigned to a game yet</h2>
                <p>Your teacher hasn’t assigned you to a game or nation yet.</p>
                <p>Please contact <strong><?php echo htmlspecialchars($teacher_email); ?></strong> for details.</p>
            </div>
        </main>
    </div>
    </body>
    </html>
    <?php
    exit;
}

// --- Load teammates for this nation -----------------------------------------
$teammates = [];
if ($nation_id > 0) {
    $stmt = $pdo->prepare(
        'SELECT
             s.id          AS student_id,
             s.label       AS student_label,
             nm.role_label,
             nm.privileges_json
         FROM nation_members nm
         JOIN students s ON s.id = nm.student_id
         WHERE nm.nation_id = ?
         ORDER BY s.created_at ASC, s.id ASC'
    );
    $stmt->execute([$nation_id]);
    $teammates = $stmt->fetchAll();
}

// --- Flag data ---------------------------------------------------------------
$primary   = $data['primary_color']   ?: '#0044aa';
$secondary = $data['secondary_color'] ?: null;
$stripe    = $data['stripe_style']    ?: 'none';
$width_f   = $data['stripe_width_fraction'] ?: '1/2';
if (!$secondary || $stripe === 'none') {
    $stripe = 'none';
}

// --- Turn / timing info ------------------------------------------------------
$current_turn = isset($data['current_turn']) ? (int)$data['current_turn'] : 1;
$turn_mode    = $data['turn_mode']    ?? 'manual';
$next_turn_at = $data['next_turn_at'] ?? null;
?>
<!doctype html>
<html>
<head>
    <meta charset="utf-8">
    <title>A Possible World – <?php echo htmlspecialchars($game_name); ?></title>
    <link rel="stylesheet" href="style.css">
    <style>
        .turn-bar {
            padding: 0.5rem 0.75rem;
            background: #1b2838;
            color: #f5f5f5;
            border-radius: 0.4rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 0.9rem;
            margin-bottom: 1rem;
        }
        .turn-bar span.label {
            font-weight: 600;
        }
        .game-tabs {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 0.75rem;
            border-bottom: 1px solid rgba(0,0,0,0.1);
            flex-wrap: wrap;
        }
        .game-tabs button {
            border: none;
            background: transparent;
            padding: 0.4rem 0.8rem;
            border-radius: 0.4rem 0.4rem 0 0;
            cursor: pointer;
            font-size: 0.9rem;
        }
        .game-tabs button.active {
            background: #ffffff;
            border: 1px solid rgba(0,0,0,0.1);
            border-bottom-color: #ffffff;
        }
        .game-tab-pane {
            display: none;
        }
        .game-tab-pane.active {
            display: block;
        }
        .flag-preview-large {
            width: 260px;
            height: 156px;
            border-radius: 6px;
            border: 1px solid rgba(0,0,0,0.3);
            overflow: hidden;
        }
        .flag-preview-stripes {
            width: 100%;
            height: 100%;
        }
        .gov-grid {
            display: grid;
            grid-template-columns: minmax(0, 2fr) minmax(260px, 260px);
            gap: 1rem;
        }
        @media (max-width: 800px) {
            .gov-grid {
                grid-template-columns: 1fr;
            }
        }
        .gov-meta-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 0.9rem;
        }
        .gov-meta-table th,
        .gov-meta-table td {
            padding: 0.2rem 0.4rem;
            text-align: left;
        }
        .gov-meta-table th {
            width: 35%;
            font-weight: 600;
        }
        .team-table {
            width: 100%;
            border-collapse: collapse;
            font-size: 0.9rem;
            margin-top: 0.5rem;
        }
        .team-table th,
        .team-table td {
            padding: 0.2rem 0.4rem;
            text-align: left;
        }
        .team-table th {
            border-bottom: 1px solid rgba(0,0,0,0.1);
        }
        .teacher-contact {
            font-size: 0.9rem;
            margin-top: 0.3rem;
        }
        .teacher-contact a {
            color: inherit;
            text-decoration: underline;
        }
    </style>
    <script>
        function switchGameTab(id) {
            document.querySelectorAll('.game-tab-pane').forEach(p => p.classList.remove('active'));
            document.querySelectorAll('.game-tabs button').forEach(b => b.classList.remove('active'));
            const pane = document.getElementById('game-tab-' + id);
            const btn  = document.getElementById('btn-tab-' + id);
            if (pane) pane.classList.add('active');
            if (btn)  btn.classList.add('active');
        }

        function startTurnCountdown() {
            const bar = document.getElementById('turn-bar');
            if (!bar) return;

            const mode = bar.dataset.mode || 'manual';
            const next = bar.dataset.nextturn || '';

            const countdownEl = document.getElementById('turn-countdown');
            if (!countdownEl || mode === 'manual' || !next) {
                if (countdownEl && mode === 'manual') {
                    countdownEl.textContent = 'Turn advances when your teacher decides.';
                }
                return;
            }

            const target = new Date(next.replace(' ', 'T'));

            function update() {
                const now = new Date();
                const diff = target - now;
                if (diff <= 0) {
                    countdownEl.textContent = 'Next turn is starting now (or soon)…';
                    return;
                }
                const totalSeconds = Math.floor(diff / 1000);
                const days = Math.floor(totalSeconds / 86400);
                const hours = Math.floor((totalSeconds % 86400) / 3600);
                const minutes = Math.floor((totalSeconds % 3600) / 60);
                const seconds = totalSeconds % 60;

                let parts = [];
                if (days) parts.push(days + 'd');
                if (hours || days) parts.push(hours + 'h');
                parts.push(String(minutes).padStart(2,'0') + 'm');
                parts.push(String(seconds).padStart(2,'0') + 's');

                countdownEl.textContent = parts.join(' ');
            }

            update();
            setInterval(update, 1000);
        }

        // Dynamic flag stripes: convert fraction to pixel stripe sizes
        function setupFlags() {
            const els = document.querySelectorAll('.flag-preview-stripes');
            els.forEach(el => {
                const primary = el.dataset.primary || '#0044aa';
                const secondary = el.dataset.secondary || '';
                const style = el.dataset.stripe || 'none';
                const fraction = el.dataset.fraction || '1/2';

                if (!secondary || style === 'none') {
                    el.style.background = primary;
                    el.style.backgroundImage = 'none';
                    return;
                }

                const parts = fraction.split('/');
                let num = parseInt(parts[0] || '1', 10);
                let den = parseInt(parts[1] || '2', 10);
                if (!den || den <= 0) den = 2;
                if (!num || num <= 0) num = 1;

                // Use main dimension depending on stripe orientation
                const w = el.clientWidth  || 200;
                const h = el.clientHeight || 120;
                let base;
                if (style === 'horizontal') base = h;
                else if (style === 'vertical') base = w;
                else base = Math.min(w, h); // diagonal

                // fraction is "num/den" of the flag dimension – single stripe thickness:
                let single = Math.max(2, Math.round(base * num / den));
                let double = single * 2;

                let angle = 'to right';
                if (style === 'horizontal') angle = 'to bottom';
                else if (style === 'vertical') angle = 'to right';
                else if (style === 'diagonal') angle = '45deg';

                el.style.backgroundImage =
                    `repeating-linear-gradient(${angle}, ` +
                    `${primary}, ${primary} ${single}px, ` +
                    `${secondary} ${single}px, ${secondary} ${double}px)`;
            });
        }

        function setupAll() {
            switchGameTab('government'); // default
            startTurnCountdown();
            setupFlags();
            window.addEventListener('resize', function () {
                // re-calc stripes if flag size changes
                setupFlags();
            });
        }

        window.addEventListener('DOMContentLoaded', setupAll);
    </script>
</head>
<body class="app-body">
<div class="app-shell">
    <header class="app-header">
        <div>
            <div class="app-title">
                Hi, <?php echo htmlspecialchars($student_label); ?>!
                Welcome to <?php echo htmlspecialchars($game_name); ?>.
            </div>
            <div class="teacher-contact">
                Send message to <strong><?php echo htmlspecialchars($teacher_email); ?></strong>
                (your teacher can decide what “send message” means in practice).
            </div>
        </div>
    </header>

    <main class="app-main">
        <div id="turn-bar"
             class="turn-bar"
             data-mode="<?php echo htmlspecialchars($turn_mode); ?>"
             data-nextturn="<?php echo htmlspecialchars($next_turn_at ?? ''); ?>">
            <span class="label">
                Turn <?php echo max(1, $current_turn); ?>
                &nbsp;·&nbsp;
                Mode: <?php echo htmlspecialchars(ucfirst($turn_mode)); ?>
            </span>
            <span id="turn-countdown"></span>
        </div>

        <div class="game-tabs">
            <button id="btn-tab-government" class="active" onclick="switchGameTab('government')">Government</button>
            <button id="btn-tab-map" onclick="switchGameTab('map')">Map</button>
            <button id="btn-tab-economy" onclick="switchGameTab('economy')">Economy</button>
            <button id="btn-tab-budget" onclick="switchGameTab('budget')">Budget</button>
            <button id="btn-tab-military" onclick="switchGameTab('military')">Military</button>
            <button id="btn-tab-technology" onclick="switchGameTab('technology')">Technology</button>
            <button id="btn-tab-environment" onclick="switchGameTab('environment')">Environment</button>
            <button id="btn-tab-trade" onclick="switchGameTab('trade')">Trade</button>
            <button id="btn-tab-news" onclick="switchGameTab('news')">News</button>
        </div>

        <?php
        // Make these available to partials
        $gov_data      = $data;
        $gov_teammates = $teammates;
        $gov_primary   = $primary;
        $gov_secondary = $secondary;
        $gov_stripe    = $stripe;
        $gov_width_f   = $width_f;
        $apw_game_state = $game_state;
        $apw_nation_state = $nation_state;

        include __DIR__ . '/game_tab_government.php';
        include __DIR__ . '/game_tab_map.php';
        include __DIR__ . '/game_tab_economy.php';
        include __DIR__ . '/game_tab_budget.php';
        include __DIR__ . '/game_tab_military.php';
        include __DIR__ . '/game_tab_technology.php';
        include __DIR__ . '/game_tab_environment.php';
        include __DIR__ . '/game_tab_trade.php';
        include __DIR__ . '/game_tab_news.php';
        ?>
    </main>
</div>
</body>
</html>
