"""Utility to zip the current repository for download.

Run this script locally to generate a zip file of the repo, matching the
snippet the user previously ran in the notebook:

>>> import shutil
>>> shutil.make_archive("/mnt/data/apossibleworld", "zip", "/workspace/apossibleworld")

By default we mirror that behavior and write ``/mnt/data/apossibleworld.zip``.
You can override the output path with ``--output``.
"""

import argparse
import shutil
from pathlib import Path

REPO_ROOT = Path(__file__).resolve().parent
DEFAULT_OUTPUT = Path('/mnt/data/apossibleworld.zip')


def build_archive(output: Path) -> Path:
    """Create a zip archive of the repository at the requested path."""
    output.parent.mkdir(parents=True, exist_ok=True)
    archive_base = output.with_suffix('')  # shutil adds .zip
    return Path(shutil.make_archive(str(archive_base), 'zip', str(REPO_ROOT)))


def main() -> None:
    parser = argparse.ArgumentParser(description="Zip the repository for download")
    parser.add_argument(
        "--output",
        type=Path,
        default=DEFAULT_OUTPUT,
        help=f"Destination zip path (default: {DEFAULT_OUTPUT})",
    )
    args = parser.parse_args()

    print(f"Creating archive at {args.output} ...")
    archive_path = build_archive(args.output)
    print(f"Archive created: {archive_path}")


if __name__ == "__main__":
    main()
